<?php
/**
 * Randevu Sistemi V7 - Hasta API
 * o2nar.com/test/api/patients.php
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../includes/config.php';

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getPatient($_GET['id']);
            } elseif (isset($_GET['search'])) {
                searchPatients($_GET['search']);
            } elseif (isset($_GET['phone'])) {
                findByPhone($_GET['phone']);
            } else {
                getPatients();
            }
            break;
        case 'POST':
            createPatient();
            break;
        case 'PUT':
            updatePatient();
            break;
        case 'DELETE':
            deletePatient();
            break;
        default:
            jsonResponse(['error' => 'Method not allowed'], 405);
    }
} catch (Exception $e) {
    jsonResponse(['error' => $e->getMessage()], 500);
}

// Hasta arama (randevu modal için)
function searchPatients($query) {
    $query = '%' . $query . '%';
    
    $sql = "
        SELECT id, first_name, last_name, phone, email, is_blocked
        FROM patients
        WHERE (CONCAT(first_name, ' ', last_name) LIKE ? OR phone LIKE ? OR email LIKE ?)
        AND is_blocked = 0
        ORDER BY first_name, last_name
        LIMIT 20
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute([$query, $query, $query]);
    $patients = $stmt->fetchAll();
    
    jsonResponse([
        'success' => true,
        'patients' => $patients
    ]);
}

// Telefon ile hasta bul (yeni hasta eklerken kontrol)
function findByPhone($phone) {
    $phone = normalizePhone($phone);
    
    $sql = "
        SELECT id, first_name, last_name, phone, email
        FROM patients
        WHERE REPLACE(REPLACE(phone, ' ', ''), '+', '') = REPLACE(REPLACE(?, ' ', ''), '+', '')
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute([$phone]);
    $patients = $stmt->fetchAll();
    
    jsonResponse([
        'success' => true,
        'patients' => $patients,
        'count' => count($patients)
    ]);
}

// Tüm hastaları getir (admin panel)
function getPatients() {
    $page = $_GET['page'] ?? 1;
    $limit = $_GET['limit'] ?? 50;
    $offset = ($page - 1) * $limit;
    $status = $_GET['status'] ?? 'all';
    $tag = $_GET['tag'] ?? null;
    $sort = $_GET['sort'] ?? 'name';
    $order = $_GET['order'] ?? 'asc';
    
    $where = [];
    $params = [];
    
    if ($status === 'active') {
        $where[] = "p.is_blocked = 0";
    } elseif ($status === 'blocked') {
        $where[] = "p.is_blocked = 1";
    }
    
    $whereClause = count($where) > 0 ? 'WHERE ' . implode(' AND ', $where) : '';
    
    // Sıralama
    $orderBy = match($sort) {
        'name' => 'p.first_name ' . $order . ', p.last_name ' . $order,
        'created' => 'p.created_at ' . $order,
        'last_visit' => 'p.last_appointment_at ' . $order,
        default => 'p.first_name ' . $order
    };
    
    // Toplam sayı
    $countSql = "SELECT COUNT(*) as total FROM patients p $whereClause";
    $countStmt = db()->prepare($countSql);
    $countStmt->execute($params);
    $total = $countStmt->fetch()['total'];
    
    // Hastaları getir
    $sql = "
        SELECT 
            p.*,
            (SELECT COUNT(*) FROM appointments WHERE patient_id = p.id) as total_appointments,
            (SELECT COUNT(*) FROM appointments WHERE patient_id = p.id AND status = 'completed') as completed_appointments,
            (SELECT COUNT(*) FROM appointments WHERE patient_id = p.id AND status = 'no_show') as noshow_count,
            (SELECT GROUP_CONCAT(pt.name SEPARATOR ',') FROM patient_tag_relations ptr JOIN patient_tags pt ON ptr.tag_id = pt.id WHERE ptr.patient_id = p.id) as tags
        FROM patients p
        $whereClause
        ORDER BY $orderBy
        LIMIT $limit OFFSET $offset
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute($params);
    $patients = $stmt->fetchAll();
    
    // Tags'i array'e çevir
    foreach ($patients as &$patient) {
        $patient['tags'] = $patient['tags'] ? explode(',', $patient['tags']) : [];
    }
    
    jsonResponse([
        'success' => true,
        'patients' => $patients,
        'total' => $total,
        'page' => (int)$page,
        'limit' => (int)$limit,
        'pages' => ceil($total / $limit)
    ]);
}

// Tek hasta detay
function getPatient($id) {
    $sql = "
        SELECT 
            p.*,
            (SELECT COUNT(*) FROM appointments WHERE patient_id = p.id) as total_appointments,
            (SELECT COUNT(*) FROM appointments WHERE patient_id = p.id AND status = 'completed') as completed_appointments,
            (SELECT COUNT(*) FROM appointments WHERE patient_id = p.id AND status = 'no_show') as noshow_count
        FROM patients p
        WHERE p.id = ?
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute([$id]);
    $patient = $stmt->fetch();
    
    if (!$patient) {
        jsonResponse(['error' => 'Hasta bulunamadı'], 404);
    }
    
    // Etiketler
    $tagsSql = "
        SELECT pt.* FROM patient_tags pt
        JOIN patient_tag_relations ptr ON pt.id = ptr.tag_id
        WHERE ptr.patient_id = ?
    ";
    $tagsStmt = db()->prepare($tagsSql);
    $tagsStmt->execute([$id]);
    $patient['tags'] = $tagsStmt->fetchAll();
    
    // Notlar
    $notesSql = "
        SELECT pn.*, CONCAT(s.first_name, ' ', s.last_name) as created_by_name
        FROM patient_notes pn
        LEFT JOIN staff s ON pn.created_by = s.id
        WHERE pn.patient_id = ?
        ORDER BY pn.created_at DESC
        LIMIT 20
    ";
    $notesStmt = db()->prepare($notesSql);
    $notesStmt->execute([$id]);
    $patient['notes'] = $notesStmt->fetchAll();
    
    // Randevu geçmişi
    $apptsSql = "
        SELECT 
            a.*,
            s.name as service_name,
            CONCAT(st.first_name, ' ', st.last_name) as staff_name
        FROM appointments a
        LEFT JOIN services s ON a.service_id = s.id
        LEFT JOIN staff st ON a.staff_id = st.id
        WHERE a.patient_id = ?
        ORDER BY a.appointment_date DESC, a.start_time DESC
        LIMIT 50
    ";
    $apptsStmt = db()->prepare($apptsSql);
    $apptsStmt->execute([$id]);
    $patient['appointments'] = $apptsStmt->fetchAll();
    
    // İletişim geçmişi
    $commSql = "
        SELECT * FROM patient_communications
        WHERE patient_id = ?
        ORDER BY created_at DESC
        LIMIT 30
    ";
    $commStmt = db()->prepare($commSql);
    $commStmt->execute([$id]);
    $patient['communications'] = $commStmt->fetchAll();
    
    // Aynı telefon numarasına sahip hastalar (aile)
    $familySql = "
        SELECT id, first_name, last_name, phone
        FROM patients
        WHERE phone = ? AND id != ?
    ";
    $familyStmt = db()->prepare($familySql);
    $familyStmt->execute([$patient['phone'], $id]);
    $patient['family_members'] = $familyStmt->fetchAll();
    
    jsonResponse([
        'success' => true,
        'patient' => $patient
    ]);
}

// Yeni hasta oluştur
function createPatient() {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (empty($data['first_name']) || empty($data['phone'])) {
        jsonResponse(['error' => 'Ad ve telefon zorunlu'], 400);
    }
    
    $phone = normalizePhone($data['phone']);
    
    $sql = "
        INSERT INTO patients 
        (first_name, last_name, phone, email, birth_date, gender, tc_no, address, city, notes, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute([
        $data['first_name'],
        $data['last_name'] ?? null,
        $phone,
        $data['email'] ?? null,
        $data['birth_date'] ?? null,
        $data['gender'] ?? null,
        $data['tc_no'] ?? null,
        $data['address'] ?? null,
        $data['city'] ?? null,
        $data['notes'] ?? null
    ]);
    
    $patientId = db()->lastInsertId();
    
    // Etiketler ekle
    if (!empty($data['tags']) && is_array($data['tags'])) {
        $tagStmt = db()->prepare("INSERT INTO patient_tag_relations (patient_id, tag_id) VALUES (?, ?)");
        foreach ($data['tags'] as $tagId) {
            $tagStmt->execute([$patientId, $tagId]);
        }
    }
    
    jsonResponse([
        'success' => true,
        'id' => $patientId,
        'message' => 'Hasta oluşturuldu'
    ]);
}

// Hasta güncelle
function updatePatient() {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (empty($data['id'])) {
        jsonResponse(['error' => 'ID gerekli'], 400);
    }
    
    // Kara listeye alma/çıkarma
    if (isset($data['is_blocked'])) {
        $sql = "UPDATE patients SET is_blocked = ?, block_reason = ?, updated_at = NOW() WHERE id = ?";
        $stmt = db()->prepare($sql);
        $stmt->execute([
            $data['is_blocked'] ? 1 : 0,
            $data['block_reason'] ?? null,
            $data['id']
        ]);
        
        jsonResponse(['success' => true, 'message' => $data['is_blocked'] ? 'Hasta kara listeye alındı' : 'Hasta kara listeden çıkarıldı']);
        return;
    }
    
    // Bildirim tercihleri
    if (isset($data['notifications'])) {
        $sql = "UPDATE patients SET sms_enabled = ?, whatsapp_enabled = ?, email_enabled = ?, updated_at = NOW() WHERE id = ?";
        $stmt = db()->prepare($sql);
        $stmt->execute([
            $data['notifications']['sms'] ?? 1,
            $data['notifications']['whatsapp'] ?? 1,
            $data['notifications']['email'] ?? 1,
            $data['id']
        ]);
        
        jsonResponse(['success' => true, 'message' => 'Bildirim tercihleri güncellendi']);
        return;
    }
    
    // Genel güncelleme
    $phone = normalizePhone($data['phone']);
    
    $sql = "
        UPDATE patients SET
            first_name = ?,
            last_name = ?,
            phone = ?,
            email = ?,
            birth_date = ?,
            gender = ?,
            tc_no = ?,
            address = ?,
            city = ?,
            notes = ?,
            updated_at = NOW()
        WHERE id = ?
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute([
        $data['first_name'],
        $data['last_name'] ?? null,
        $phone,
        $data['email'] ?? null,
        $data['birth_date'] ?? null,
        $data['gender'] ?? null,
        $data['tc_no'] ?? null,
        $data['address'] ?? null,
        $data['city'] ?? null,
        $data['notes'] ?? null,
        $data['id']
    ]);
    
    // Etiketleri güncelle
    if (isset($data['tags'])) {
        // Mevcut etiketleri sil
        $deleteStmt = db()->prepare("DELETE FROM patient_tag_relations WHERE patient_id = ?");
        $deleteStmt->execute([$data['id']]);
        
        // Yeni etiketleri ekle
        if (is_array($data['tags']) && count($data['tags']) > 0) {
            $tagStmt = db()->prepare("INSERT INTO patient_tag_relations (patient_id, tag_id) VALUES (?, ?)");
            foreach ($data['tags'] as $tagId) {
                $tagStmt->execute([$data['id'], $tagId]);
            }
        }
    }
    
    jsonResponse(['success' => true, 'message' => 'Hasta güncellendi']);
}

// Hasta sil (soft delete)
function deletePatient() {
    $id = $_GET['id'] ?? null;
    
    if (!$id) {
        jsonResponse(['error' => 'ID gerekli'], 400);
    }
    
    // Randevusu var mı kontrol et
    $checkStmt = db()->prepare("SELECT COUNT(*) as count FROM appointments WHERE patient_id = ?");
    $checkStmt->execute([$id]);
    $count = $checkStmt->fetch()['count'];
    
    if ($count > 0) {
        jsonResponse(['error' => 'Bu hastanın randevuları var, silinemez. Kara listeye alabilirsiniz.'], 400);
    }
    
    $stmt = db()->prepare("DELETE FROM patients WHERE id = ?");
    $stmt->execute([$id]);
    
    jsonResponse(['success' => true, 'message' => 'Hasta silindi']);
}
