<?php
/**
 * Randevu Sistemi V7 - Hasta İçe Aktarma API
 */

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../includes/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['error' => 'Method not allowed'], 405);
}

try {
    if (!isset($_FILES['file'])) {
        jsonResponse(['error' => 'Dosya yüklenmedi'], 400);
    }
    
    $file = $_FILES['file'];
    $content = file_get_contents($file['tmp_name']);
    
    // UTF-8 BOM temizle
    $content = preg_replace('/^\xEF\xBB\xBF/', '', $content);
    
    $lines = explode("\n", $content);
    $lines = array_filter($lines, fn($l) => trim($l) !== '');
    
    if (count($lines) < 2) {
        jsonResponse(['error' => 'Dosya boş veya geçersiz'], 400);
    }
    
    // Header satırını parse et
    $headerLine = array_shift($lines);
    $delimiter = strpos($headerLine, "\t") !== false ? "\t" : (strpos($headerLine, ';') !== false ? ';' : ',');
    $headers = array_map(fn($h) => strtolower(trim(str_replace(['"', "'"], '', $h))), explode($delimiter, $headerLine));
    
    // Kolon eşleştirme
    $columnMap = [
        'first_name' => ['ad', 'first_name', 'firstname', 'isim', 'ad soyad', 'adsoyad', 'name'],
        'last_name' => ['soyad', 'last_name', 'lastname', 'soyadi'],
        'phone' => ['telefon', 'phone', 'tel', 'gsm', 'cep', 'cep telefonu'],
        'email' => ['email', 'e-posta', 'eposta', 'mail'],
        'birth_date' => ['dogum_tarihi', 'birth_date', 'birthdate', 'dogum tarihi', 'doğum tarihi'],
        'gender' => ['cinsiyet', 'gender'],
        'address' => ['adres', 'address'],
        'city' => ['sehir', 'city', 'il', 'şehir'],
        'tc_no' => ['tc', 'tc_no', 'tcno', 'tc kimlik', 'kimlik no']
    ];
    
    $indexes = [];
    foreach ($columnMap as $field => $aliases) {
        foreach ($aliases as $alias) {
            $idx = array_search($alias, $headers);
            if ($idx !== false) {
                $indexes[$field] = $idx;
                break;
            }
        }
    }
    
    // En azından isim ve telefon olmalı
    if (!isset($indexes['first_name']) && !isset($indexes['phone'])) {
        jsonResponse(['error' => 'Ad veya telefon sütunu bulunamadı'], 400);
    }
    
    $imported = 0;
    $skipped = 0;
    $errors = [];
    
    $insertStmt = db()->prepare("
        INSERT INTO patients (first_name, last_name, phone, email, birth_date, gender, address, city, tc_no, source, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'import', NOW())
        ON DUPLICATE KEY UPDATE updated_at = NOW()
    ");
    
    foreach ($lines as $lineNum => $line) {
        $values = array_map(fn($v) => trim(str_replace(['"', "'"], '', $v)), explode($delimiter, $line));
        
        $firstName = $values[$indexes['first_name'] ?? -1] ?? '';
        $lastName = $values[$indexes['last_name'] ?? -1] ?? '';
        $phone = $values[$indexes['phone'] ?? -1] ?? '';
        $email = $values[$indexes['email'] ?? -1] ?? '';
        $birthDate = $values[$indexes['birth_date'] ?? -1] ?? '';
        $gender = $values[$indexes['gender'] ?? -1] ?? '';
        $address = $values[$indexes['address'] ?? -1] ?? '';
        $city = $values[$indexes['city'] ?? -1] ?? '';
        $tcNo = $values[$indexes['tc_no'] ?? -1] ?? '';
        
        // Ad soyad birleşikse ayır
        if (!$lastName && strpos($firstName, ' ') !== false) {
            $parts = explode(' ', $firstName);
            $lastName = array_pop($parts);
            $firstName = implode(' ', $parts);
        }
        
        // Telefon normalize
        if ($phone) {
            $phone = normalizePhone($phone);
        }
        
        // Cinsiyet normalize
        if ($gender) {
            $gender = strtolower($gender);
            if (in_array($gender, ['erkek', 'male', 'e', 'm'])) {
                $gender = 'male';
            } elseif (in_array($gender, ['kadın', 'kadin', 'female', 'k', 'f'])) {
                $gender = 'female';
            } else {
                $gender = null;
            }
        } else {
            $gender = null;
        }
        
        // Doğum tarihi normalize
        if ($birthDate) {
            $date = strtotime($birthDate);
            if ($date) {
                $birthDate = date('Y-m-d', $date);
            } else {
                $birthDate = null;
            }
        } else {
            $birthDate = null;
        }
        
        // Validasyon
        if (!$firstName && !$phone) {
            $skipped++;
            continue;
        }
        
        if (!$firstName) $firstName = 'İsimsiz';
        
        try {
            $insertStmt->execute([
                $firstName,
                $lastName ?: null,
                $phone ?: null,
                $email ?: null,
                $birthDate,
                $gender,
                $address ?: null,
                $city ?: null,
                $tcNo ?: null
            ]);
            $imported++;
        } catch (Exception $e) {
            $skipped++;
            $errors[] = "Satır " . ($lineNum + 2) . ": " . $e->getMessage();
        }
    }
    
    jsonResponse([
        'success' => true,
        'imported' => $imported,
        'skipped' => $skipped,
        'errors' => array_slice($errors, 0, 10),
        'message' => "$imported hasta içe aktarıldı, $skipped atlandı"
    ]);
    
} catch (Exception $e) {
    jsonResponse(['error' => $e->getMessage()], 500);
}
