<?php
/**
 * Randevu Sistemi V7 - Randevu API
 * o2nar.com/test/api/appointments.php
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../includes/config.php';

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            getAppointments();
            break;
        case 'POST':
            createAppointment();
            break;
        case 'PUT':
            updateAppointment();
            break;
        case 'DELETE':
            deleteAppointment();
            break;
        default:
            jsonResponse(['error' => 'Method not allowed'], 405);
    }
} catch (Exception $e) {
    jsonResponse(['error' => $e->getMessage()], 500);
}

// Randevuları getir
function getAppointments() {
    $date = $_GET['date'] ?? date('Y-m-d');
    $staffId = $_GET['staff_id'] ?? null;
    
    $sql = "
        SELECT 
            a.*,
            CONCAT(p.first_name, ' ', p.last_name) as patient_name,
            p.phone as patient_phone,
            p.notes as patient_notes,
            p.is_blocked as patient_blocked,
            s.name as service_name,
            s.color as service_color,
            CONCAT(st.first_name, ' ', st.last_name) as staff_name,
            st.color as staff_color
        FROM appointments a
        LEFT JOIN patients p ON a.patient_id = p.id
        LEFT JOIN services s ON a.service_id = s.id
        LEFT JOIN staff st ON a.staff_id = st.id
        WHERE a.appointment_date = ?
    ";
    
    $params = [$date];
    
    if ($staffId) {
        $sql .= " AND a.staff_id = ?";
        $params[] = $staffId;
    }
    
    $sql .= " ORDER BY a.start_time ASC";
    
    $stmt = db()->prepare($sql);
    $stmt->execute($params);
    $appointments = $stmt->fetchAll();
    
    // Blokları da getir
    $blockSql = "
        SELECT bt.*, CONCAT(st.first_name, ' ', st.last_name) as staff_name
        FROM blocked_times bt
        LEFT JOIN staff st ON bt.staff_id = st.id
        WHERE bt.block_date = ?
    ";
    $blockParams = [$date];
    
    if ($staffId) {
        $blockSql .= " AND bt.staff_id = ?";
        $blockParams[] = $staffId;
    }
    
    $blockStmt = db()->prepare($blockSql);
    $blockStmt->execute($blockParams);
    $blocks = $blockStmt->fetchAll();
    
    jsonResponse([
        'success' => true,
        'appointments' => $appointments,
        'blocks' => $blocks,
        'date' => $date
    ]);
}

// Yeni randevu oluştur
function createAppointment() {
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Yeni hasta oluştur
    if (!empty($data['new_patient'])) {
        $patient = $data['new_patient'];
        
        // Telefon kontrolü
        $phone = normalizePhone($patient['phone']);
        
        $stmt = db()->prepare("INSERT INTO patients (first_name, last_name, phone, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->execute([$patient['first_name'], $patient['last_name'], $phone]);
        $data['patient_id'] = db()->lastInsertId();
    }
    
    if (empty($data['patient_id'])) {
        jsonResponse(['error' => 'Hasta seçilmedi'], 400);
    }
    
    // Çakışma kontrolü
    $conflictSql = "
        SELECT id FROM appointments 
        WHERE staff_id = ? AND appointment_date = ? 
        AND status NOT IN ('cancelled')
        AND (
            (start_time < ? AND end_time > ?) OR
            (start_time < ? AND end_time > ?) OR
            (start_time >= ? AND end_time <= ?)
        )
    ";
    $conflictStmt = db()->prepare($conflictSql);
    $conflictStmt->execute([
        $data['staff_id'],
        $data['appointment_date'],
        $data['end_time'], $data['start_time'],
        $data['end_time'], $data['end_time'],
        $data['start_time'], $data['end_time']
    ]);
    
    if ($conflictStmt->fetch()) {
        jsonResponse(['error' => 'Bu saatte başka bir randevu var'], 400);
    }
    
    // Süreyi hesapla
    $startMins = strtotime($data['start_time']) / 60;
    $endMins = strtotime($data['end_time']) / 60;
    $duration = ($endMins - $startMins);
    
    $sql = "
        INSERT INTO appointments 
        (patient_id, staff_id, service_id, appointment_date, start_time, end_time, duration, status, notes, source, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'admin', NOW())
    ";
    
    $stmt = db()->prepare($sql);
    $stmt->execute([
        $data['patient_id'],
        $data['staff_id'],
        $data['service_id'],
        $data['appointment_date'],
        $data['start_time'],
        $data['end_time'],
        $duration,
        $data['status'] ?? 'pending',
        $data['notes'] ?? null
    ]);
    
    $appointmentId = db()->lastInsertId();
    
    // Hasta son randevu bilgisini güncelle
    $updatePatient = db()->prepare("UPDATE patients SET last_appointment_at = ?, total_appointments = total_appointments + 1 WHERE id = ?");
    $updatePatient->execute([$data['appointment_date'] . ' ' . $data['start_time'], $data['patient_id']]);
    
    jsonResponse([
        'success' => true,
        'id' => $appointmentId,
        'message' => 'Randevu oluşturuldu'
    ]);
}

// Randevu güncelle
function updateAppointment() {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (empty($data['id'])) {
        jsonResponse(['error' => 'ID gerekli'], 400);
    }
    
    // Sadece zaman güncelleme (drag & drop / resize)
    if (isset($data['start_time']) && isset($data['end_time']) && !isset($data['patient_id'])) {
        // Süreyi hesapla
        $startMins = strtotime($data['start_time']) / 60;
        $endMins = strtotime($data['end_time']) / 60;
        $duration = ($endMins - $startMins);
        
        $sql = "UPDATE appointments SET start_time = ?, end_time = ?, duration = ?, updated_at = NOW() WHERE id = ?";
        $stmt = db()->prepare($sql);
        $stmt->execute([$data['start_time'], $data['end_time'], $duration, $data['id']]);
        
        jsonResponse(['success' => true, 'message' => 'Zaman güncellendi']);
        return;
    }
    
    // Tam güncelleme
    $sql = "
        UPDATE appointments SET
            patient_id = ?,
            staff_id = ?,
            service_id = ?,
            appointment_date = ?,
            start_time = ?,
            end_time = ?,
            status = ?,
            notes = ?,
            updated_at = NOW()
        WHERE id = ?
    ";
    
    // Süreyi hesapla
    $startMins = strtotime($data['start_time']) / 60;
    $endMins = strtotime($data['end_time']) / 60;
    $duration = ($endMins - $startMins);
    
    $stmt = db()->prepare($sql);
    $stmt->execute([
        $data['patient_id'],
        $data['staff_id'],
        $data['service_id'],
        $data['appointment_date'],
        $data['start_time'],
        $data['end_time'],
        $data['status'],
        $data['notes'] ?? null,
        $data['id']
    ]);
    
    jsonResponse(['success' => true, 'message' => 'Randevu güncellendi']);
}

// Randevu sil
function deleteAppointment() {
    $id = $_GET['id'] ?? null;
    
    if (!$id) {
        jsonResponse(['error' => 'ID gerekli'], 400);
    }
    
    // Soft delete - durumu iptal yap
    $stmt = db()->prepare("UPDATE appointments SET status = 'cancelled', cancelled_at = NOW() WHERE id = ?");
    $stmt->execute([$id]);
    
    jsonResponse(['success' => true, 'message' => 'Randevu iptal edildi']);
}
