<?php
/**
 * Randevu Sistemi V7 - Hasta Yönetim Paneli
 * o2nar.com/test/admin/patients.php
 */

require_once __DIR__ . '/../includes/auth.php';
requireAdmin();

$currentUser = currentUser();

// Etiketler
$tagsStmt = db()->query("SELECT * FROM patient_tags ORDER BY name");
$tags = $tagsStmt->fetchAll();

$pageTitle = 'Hasta Yönetimi';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?> - <?= CLINIC_NAME ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .modal { display: none; }
        .modal.active { display: flex; }
        .tag-badge {
            display: inline-flex;
            align-items: center;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 500;
        }
        .checkbox-col { width: 40px; }
        .actions-col { width: 100px; }
        tbody tr:hover { background: #f8fafc; }
        .selected-row { background: #eff6ff !important; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    <!-- Header -->
    <header class="bg-white border-b border-gray-200 sticky top-0 z-40">
        <div class="flex items-center justify-between px-4 py-3">
            <div class="flex items-center gap-4">
                <a href="index.php" class="text-gray-600 hover:text-gray-800">← Takvim</a>
                <h1 class="text-lg font-bold text-gray-800">👥 Hasta Yönetimi</h1>
            </div>
            <div class="flex items-center gap-2">
                <span class="text-sm text-gray-600"><?= e($currentUser['name']) ?></span>
                <a href="logout.php" class="text-sm text-red-600 hover:text-red-800">Çıkış</a>
            </div>
        </div>
    </header>

    <div class="p-4">
        <!-- Üst Toolbar -->
        <div class="bg-white rounded-lg shadow-sm p-4 mb-4">
            <div class="flex flex-wrap items-center gap-3">
                <!-- Arama -->
                <div class="flex-1 min-w-[250px]">
                    <input type="text" id="searchInput" placeholder="🔍 İsim, telefon, e-posta ara..." 
                           onkeyup="debounceSearch()" class="w-full px-4 py-2 border rounded-lg">
                </div>
                
                <!-- Etiket Filtre -->
                <select id="tagFilter" onchange="loadPatients()" class="px-3 py-2 border rounded-lg">
                    <option value="">Tüm Etiketler</option>
                    <?php foreach ($tags as $tag): ?>
                    <option value="<?= $tag['id'] ?>"><?= e($tag['name']) ?></option>
                    <?php endforeach; ?>
                </select>
                
                <!-- Durum Filtre -->
                <select id="statusFilter" onchange="loadPatients()" class="px-3 py-2 border rounded-lg">
                    <option value="all">Tüm Durumlar</option>
                    <option value="active">Aktif</option>
                    <option value="blocked">Kara Liste</option>
                </select>
                
                <!-- Sütun Ayarları -->
                <div class="relative">
                    <button onclick="toggleColumnSettings()" class="px-3 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg text-sm">⚙️ Sütunlar</button>
                    <div id="columnSettings" class="hidden absolute right-0 top-full mt-1 bg-white rounded-lg shadow-xl p-3 z-50 min-w-[180px]">
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" checked onchange="toggleColumn('phone')" id="colPhone"> Telefon</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" checked onchange="toggleColumn('email')" id="colEmail"> E-posta</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" checked onchange="toggleColumn('age')" id="colAge"> Yaş</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" checked onchange="toggleColumn('tags')" id="colTags"> Etiketler</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" checked onchange="toggleColumn('status')" id="colStatus"> Durum</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" onchange="toggleColumn('source')" id="colSource"> Kaynak</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" checked onchange="toggleColumn('lastVisit')" id="colLastVisit"> Son Ziyaret</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" onchange="toggleColumn('notifications')" id="colNotifications"> Bildirimler</label>
                        <label class="flex items-center gap-2 py-1 cursor-pointer"><input type="checkbox" onchange="toggleColumn('created')" id="colCreated"> Kayıt Tarihi</label>
                    </div>
                </div>
                
                <!-- Butonlar -->
                <button onclick="openImportModal()" class="px-3 py-2 bg-blue-100 text-blue-700 hover:bg-blue-200 rounded-lg text-sm font-medium">📤 İçe Aktar</button>
                <button onclick="exportPatients()" class="px-3 py-2 bg-green-100 text-green-700 hover:bg-green-200 rounded-lg text-sm font-medium">📥 Excel</button>
                <button onclick="openAddPatientModal()" class="px-3 py-2 bg-blue-600 text-white hover:bg-blue-700 rounded-lg text-sm font-medium">➕ Yeni Hasta</button>
            </div>
        </div>

        <!-- İstatistikler -->
        <div class="flex gap-4 mb-4">
            <div class="bg-white rounded-lg shadow-sm px-4 py-3">
                <span id="totalCount" class="text-xl font-bold text-blue-600">0</span>
                <span class="text-gray-500 text-sm ml-1">hasta</span>
            </div>
            <div class="bg-white rounded-lg shadow-sm px-4 py-3">
                <span id="activeCount" class="text-xl font-bold text-green-600">0</span>
                <span class="text-gray-500 text-sm ml-1">aktif</span>
            </div>
            <div class="bg-white rounded-lg shadow-sm px-4 py-3">
                <span id="blockedCount" class="text-xl font-bold text-red-600">0</span>
                <span class="text-gray-500 text-sm ml-1">kara liste</span>
            </div>
            <div id="selectedStats" class="bg-white rounded-lg shadow-sm px-4 py-3 hidden">
                <span id="selectedCount" class="text-xl font-bold text-purple-600">0</span>
                <span class="text-gray-500 text-sm ml-1">seçili</span>
            </div>
        </div>

        <!-- Toplu İşlemler -->
        <div id="bulkActions" class="hidden bg-blue-50 rounded-lg p-3 mb-4 flex items-center gap-3">
            <span class="text-sm text-blue-700 font-medium">Toplu İşlemler:</span>
            <button onclick="bulkAddTag()" class="px-3 py-1.5 bg-blue-100 text-blue-700 hover:bg-blue-200 rounded text-sm">🏷️ Etiket Ekle</button>
            <button onclick="bulkEnableSms()" class="px-3 py-1.5 bg-green-100 text-green-700 hover:bg-green-200 rounded text-sm">📱 SMS Aç</button>
            <button onclick="bulkDisableSms()" class="px-3 py-1.5 bg-amber-100 text-amber-700 hover:bg-amber-200 rounded text-sm">🔕 SMS Kapat</button>
            <button onclick="bulkBlock()" class="px-3 py-1.5 bg-red-100 text-red-700 hover:bg-red-200 rounded text-sm">🚫 Kara Listeye Al</button>
            <button onclick="bulkUnblock()" class="px-3 py-1.5 bg-green-100 text-green-700 hover:bg-green-200 rounded text-sm">✅ Aktif Yap</button>
        </div>

        <!-- Tablo -->
        <div class="bg-white rounded-lg shadow-sm overflow-hidden">
            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-50 sticky top-0">
                        <tr>
                            <th class="checkbox-col px-3 py-3 text-left">
                                <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                            </th>
                            <th class="px-3 py-3 text-left font-medium cursor-pointer hover:bg-gray-100" onclick="sortBy('name')">
                                Ad Soyad <span id="sortName"></span>
                            </th>
                            <th class="px-3 py-3 text-left font-medium col-phone">Telefon</th>
                            <th class="px-3 py-3 text-left font-medium col-email">E-posta</th>
                            <th class="px-3 py-3 text-center font-medium col-age">Yaş</th>
                            <th class="px-3 py-3 text-left font-medium col-tags">Etiketler</th>
                            <th class="px-3 py-3 text-left font-medium col-status">Durum</th>
                            <th class="px-3 py-3 text-left font-medium col-source hidden">Kaynak</th>
                            <th class="px-3 py-3 text-left font-medium col-lastVisit cursor-pointer hover:bg-gray-100" onclick="sortBy('last_visit')">
                                Son Ziyaret <span id="sortLastVisit"></span>
                            </th>
                            <th class="px-3 py-3 text-center font-medium col-notifications hidden">Bildirim</th>
                            <th class="px-3 py-3 text-left font-medium col-created hidden cursor-pointer hover:bg-gray-100" onclick="sortBy('created')">
                                Kayıt <span id="sortCreated"></span>
                            </th>
                            <th class="actions-col px-3 py-3 text-center font-medium">İşlem</th>
                        </tr>
                    </thead>
                    <tbody id="patientsTableBody">
                        <!-- JavaScript ile doldurulacak -->
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <div class="flex items-center justify-between px-4 py-3 border-t">
                <div class="text-sm text-gray-500">
                    <span id="showingText">0-0 / 0</span> hasta gösteriliyor
                </div>
                <div class="flex gap-1" id="pagination">
                    <!-- JavaScript ile doldurulacak -->
                </div>
            </div>
        </div>
    </div>

    <!-- Hasta Detay Modal -->
    <div id="patientModal" class="modal fixed inset-0 bg-black/50 items-center justify-center z-50">
        <div class="bg-white rounded-xl w-full max-w-3xl max-h-[90vh] overflow-auto shadow-2xl">
            <div class="sticky top-0 bg-white border-b px-6 py-4 flex justify-between items-center">
                <h2 id="patientModalTitle" class="text-lg font-semibold">Hasta Detayı</h2>
                <button onclick="closeModal('patientModal')" class="text-gray-400 hover:text-gray-600 text-2xl">&times;</button>
            </div>
            
            <div class="p-6">
                <!-- Kara Liste Uyarısı -->
                <div id="blockedWarning" class="hidden bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                    <div class="font-semibold text-red-700 mb-1">🚫 Kara Listede</div>
                    <div class="text-sm text-red-600" id="blockReason">Sebep: -</div>
                    <button onclick="unblockPatient()" class="mt-2 px-3 py-1.5 bg-green-500 text-white rounded text-sm hover:bg-green-600">🔓 Kara Listeden Çıkar</button>
                </div>
                
                <!-- Sekmeler -->
                <div class="flex gap-1 mb-4 border-b">
                    <button onclick="setPatientTab('info')" id="tabInfo" class="px-4 py-2 font-medium border-b-2 border-blue-600 text-blue-600">👤 Bilgiler</button>
                    <button onclick="setPatientTab('notes')" id="tabNotes" class="px-4 py-2 font-medium text-gray-500">📝 Notlar</button>
                    <button onclick="setPatientTab('appointments')" id="tabAppointments" class="px-4 py-2 font-medium text-gray-500">📅 Randevular</button>
                    <button onclick="setPatientTab('communication')" id="tabCommunication" class="px-4 py-2 font-medium text-gray-500">💬 İletişim</button>
                </div>
                
                <!-- Bilgiler Tab -->
                <div id="tabContentInfo">
                    <form id="patientForm">
                        <input type="hidden" id="patientId" name="id">
                        
                        <div class="grid grid-cols-2 gap-4 mb-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Ad *</label>
                                <input type="text" id="firstName" name="first_name" required class="w-full px-3 py-2 border rounded-lg">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Soyad</label>
                                <input type="text" id="lastName" name="last_name" class="w-full px-3 py-2 border rounded-lg">
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4 mb-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Telefon *</label>
                                <input type="tel" id="phone" name="phone" required class="w-full px-3 py-2 border rounded-lg">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">E-posta</label>
                                <input type="email" id="email" name="email" class="w-full px-3 py-2 border rounded-lg">
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-3 gap-4 mb-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Doğum Tarihi</label>
                                <input type="date" id="birthDate" name="birth_date" class="w-full px-3 py-2 border rounded-lg">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Cinsiyet</label>
                                <select id="gender" name="gender" class="w-full px-3 py-2 border rounded-lg">
                                    <option value="">Seçiniz</option>
                                    <option value="male">Erkek</option>
                                    <option value="female">Kadın</option>
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">TC Kimlik</label>
                                <input type="text" id="tcNo" name="tc_no" maxlength="11" class="w-full px-3 py-2 border rounded-lg">
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-1">Adres</label>
                            <input type="text" id="address" name="address" class="w-full px-3 py-2 border rounded-lg">
                        </div>
                        
                        <!-- Etiketler -->
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Etiketler</label>
                            <div id="patientTags" class="flex flex-wrap gap-2 mb-2"></div>
                            <div class="flex flex-wrap gap-2">
                                <?php foreach ($tags as $tag): ?>
                                <button type="button" onclick="togglePatientTag(<?= $tag['id'] ?>, '<?= e($tag['name']) ?>', '<?= e($tag['color']) ?>')" 
                                        class="tag-btn px-3 py-1 text-sm border border-dashed rounded-full hover:bg-gray-50" 
                                        data-tag-id="<?= $tag['id'] ?>" style="border-color: <?= e($tag['color']) ?>; color: <?= e($tag['color']) ?>">
                                    + <?= e($tag['name']) ?>
                                </button>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <!-- Bildirim Tercihleri -->
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Bildirim Tercihleri</label>
                            <div class="flex gap-4">
                                <label class="flex items-center gap-2 cursor-pointer">
                                    <input type="checkbox" id="smsEnabled" name="sms_enabled" class="w-4 h-4">
                                    <span>📱 SMS</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer">
                                    <input type="checkbox" id="whatsappEnabled" name="whatsapp_enabled" class="w-4 h-4">
                                    <span>💬 WhatsApp</span>
                                </label>
                                <label class="flex items-center gap-2 cursor-pointer">
                                    <input type="checkbox" id="emailEnabled" name="email_enabled" class="w-4 h-4">
                                    <span>📧 E-posta</span>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Genel Not -->
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-1">Genel Not</label>
                            <textarea id="patientNotes" name="notes" rows="2" class="w-full px-3 py-2 border rounded-lg"></textarea>
                        </div>
                        
                        <div class="flex gap-2 justify-between">
                            <button type="button" onclick="blockPatient()" id="blockBtn" class="px-4 py-2 bg-red-100 text-red-700 hover:bg-red-200 rounded-lg">🚫 Kara Listeye Al</button>
                            <div class="flex gap-2">
                                <button type="button" onclick="closeModal('patientModal')" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg">İptal</button>
                                <button type="submit" class="px-4 py-2 bg-blue-600 text-white hover:bg-blue-700 rounded-lg font-medium">💾 Kaydet</button>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Notlar Tab -->
                <div id="tabContentNotes" class="hidden">
                    <!-- Yeni Not -->
                    <div class="bg-gray-50 rounded-lg p-4 mb-4">
                        <h4 class="font-medium mb-3">➕ Yeni Not Ekle</h4>
                        <div class="flex gap-2 mb-2 flex-wrap">
                            <button type="button" onclick="setNoteType('general')" class="note-type-btn px-3 py-1.5 bg-blue-100 text-blue-700 rounded text-sm" data-type="general">📝 Genel</button>
                            <button type="button" onclick="setNoteType('medical')" class="note-type-btn px-3 py-1.5 bg-gray-100 text-gray-700 rounded text-sm" data-type="medical">🏥 Tıbbi</button>
                            <button type="button" onclick="setNoteType('billing')" class="note-type-btn px-3 py-1.5 bg-gray-100 text-gray-700 rounded text-sm" data-type="billing">💰 Ödeme</button>
                            <button type="button" onclick="setNoteType('complaint')" class="note-type-btn px-3 py-1.5 bg-gray-100 text-gray-700 rounded text-sm" data-type="complaint">⚠️ Şikayet</button>
                            <button type="button" onclick="setNoteType('allergy')" class="note-type-btn px-3 py-1.5 bg-gray-100 text-gray-700 rounded text-sm" data-type="allergy">🚨 Alerji</button>
                        </div>
                        <textarea id="newNoteText" rows="2" placeholder="Not yazın..." class="w-full px-3 py-2 border rounded-lg mb-2"></textarea>
                        <button type="button" onclick="addNote()" class="px-4 py-2 bg-blue-600 text-white rounded-lg text-sm hover:bg-blue-700">📝 Not Ekle</button>
                    </div>
                    
                    <!-- Mevcut Notlar -->
                    <div id="notesList" class="space-y-2 max-h-[300px] overflow-auto">
                        <!-- JavaScript ile doldurulacak -->
                    </div>
                </div>
                
                <!-- Randevular Tab -->
                <div id="tabContentAppointments" class="hidden">
                    <div id="appointmentsList" class="space-y-2 max-h-[400px] overflow-auto">
                        <!-- JavaScript ile doldurulacak -->
                    </div>
                </div>
                
                <!-- İletişim Tab -->
                <div id="tabContentCommunication" class="hidden">
                    <div id="communicationList" class="space-y-2 max-h-[400px] overflow-auto">
                        <!-- JavaScript ile doldurulacak -->
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- İçe Aktarma Modal -->
    <div id="importModal" class="modal fixed inset-0 bg-black/50 items-center justify-center z-50">
        <div class="bg-white rounded-xl p-6 w-full max-w-2xl shadow-2xl">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-lg font-semibold">📤 Hasta İçe Aktar</h2>
                <button onclick="closeModal('importModal')" class="text-gray-400 hover:text-gray-600 text-2xl">&times;</button>
            </div>
            
            <div class="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center mb-4">
                <input type="file" id="importFile" accept=".csv,.xls,.xlsx" onchange="previewImport(this)" class="hidden">
                <button onclick="document.getElementById('importFile').click()" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">📁 CSV/Excel Dosyası Seç</button>
                <p class="text-sm text-gray-500 mt-2">CSV veya Excel dosyası yükleyin</p>
            </div>
            
            <div id="importPreview" class="hidden">
                <h4 class="font-medium mb-2">Önizleme (<span id="importCount">0</span> satır)</h4>
                <div class="overflow-auto max-h-[200px] border rounded-lg mb-4">
                    <table class="w-full text-xs">
                        <thead class="bg-gray-50 sticky top-0">
                            <tr id="importHeaders"></tr>
                        </thead>
                        <tbody id="importBody"></tbody>
                    </table>
                </div>
                <div class="flex gap-2 justify-end">
                    <button onclick="closeModal('importModal')" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg">İptal</button>
                    <button onclick="executeImport()" class="px-4 py-2 bg-green-600 text-white hover:bg-green-700 rounded-lg font-medium">✅ İçe Aktar</button>
                </div>
            </div>
            
            <div class="bg-amber-50 rounded-lg p-4 mt-4">
                <h4 class="font-medium text-amber-800 mb-1">📋 CSV Formatı</h4>
                <p class="text-sm text-amber-700">Zorunlu: <strong>ad</strong> (veya first_name), <strong>telefon</strong> (veya phone)</p>
                <p class="text-sm text-amber-700">Opsiyonel: soyad, email, dogum_tarihi, cinsiyet, adres</p>
            </div>
        </div>
    </div>

    <script>
    const API_URL = '<?= API_URL ?>';
    const tags = <?= json_encode($tags) ?>;
    
    let state = {
        patients: [],
        total: 0,
        page: 1,
        limit: 50,
        sort: 'name',
        order: 'asc',
        search: '',
        selectedPatients: [],
        currentPatient: null,
        currentPatientTab: 'info',
        currentNoteType: 'general',
        importData: null
    };

    // Sayfa yüklendiğinde
    document.addEventListener('DOMContentLoaded', function() {
        loadPatients();
        
        document.getElementById('patientForm').onsubmit = savePatient;
    });

    // Debounce arama
    let searchTimeout;
    function debounceSearch() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => {
            state.search = document.getElementById('searchInput').value;
            state.page = 1;
            loadPatients();
        }, 300);
    }

    // Hastaları yükle
    async function loadPatients() {
        const tag = document.getElementById('tagFilter').value;
        const status = document.getElementById('statusFilter').value;
        
        let url = `${API_URL}/patients.php?page=${state.page}&limit=${state.limit}&sort=${state.sort}&order=${state.order}&status=${status}`;
        if (state.search) url += `&search=${encodeURIComponent(state.search)}`;
        if (tag) url += `&tag=${tag}`;
        
        try {
            const response = await fetch(url);
            const data = await response.json();
            
            if (data.success) {
                state.patients = data.patients;
                state.total = data.total;
                renderPatients();
                updateStats();
                renderPagination(data.pages);
            }
        } catch (error) {
            console.error('Yükleme hatası:', error);
        }
    }

    // Hastaları render et
    function renderPatients() {
        const tbody = document.getElementById('patientsTableBody');
        
        if (state.patients.length === 0) {
            tbody.innerHTML = '<tr><td colspan="12" class="text-center py-8 text-gray-500">Hasta bulunamadı</td></tr>';
            return;
        }
        
        tbody.innerHTML = state.patients.map(p => {
            const age = p.birth_date ? calculateAge(p.birth_date) : '-';
            const tags = p.tags ? p.tags.map(t => `<span class="tag-badge" style="background:${getTagColor(t)}20;color:${getTagColor(t)}">${t}</span>`).join(' ') : '';
            const isSelected = state.selectedPatients.includes(p.id);
            const isBlocked = p.is_blocked == 1;
            
            return `
                <tr class="${isSelected ? 'selected-row' : ''} ${isBlocked ? 'bg-red-50' : ''}" data-id="${p.id}">
                    <td class="px-3 py-2"><input type="checkbox" ${isSelected ? 'checked' : ''} onchange="toggleSelect(${p.id})"></td>
                    <td class="px-3 py-2">
                        <div class="font-medium">${p.first_name} ${p.last_name || ''}</div>
                        ${p.family_count > 1 ? '<span class="text-xs bg-indigo-100 text-indigo-700 px-1.5 py-0.5 rounded">👨‍👩‍👧</span>' : ''}
                    </td>
                    <td class="px-3 py-2 text-gray-600 col-phone">${p.phone || '-'}</td>
                    <td class="px-3 py-2 text-gray-600 col-email">${p.email || '-'}</td>
                    <td class="px-3 py-2 text-center col-age">${age}</td>
                    <td class="px-3 py-2 col-tags"><div class="flex gap-1 flex-wrap">${tags}</div></td>
                    <td class="px-3 py-2 col-status">
                        ${isBlocked 
                            ? '<span class="px-2 py-0.5 bg-red-100 text-red-700 rounded-full text-xs font-medium">🚫 Kara Liste</span>'
                            : '<span class="px-2 py-0.5 bg-green-100 text-green-700 rounded-full text-xs font-medium">Aktif</span>'
                        }
                    </td>
                    <td class="px-3 py-2 text-gray-500 text-xs col-source hidden">${p.source || '-'}</td>
                    <td class="px-3 py-2 text-gray-500 text-xs col-lastVisit">${p.last_appointment_at ? formatDate(p.last_appointment_at) : '-'}</td>
                    <td class="px-3 py-2 text-center col-notifications hidden">
                        <span class="${p.sms_enabled ? '' : 'opacity-30'}">📱</span>
                        <span class="${p.whatsapp_enabled ? '' : 'opacity-30'}">💬</span>
                        <span class="${p.email_enabled ? '' : 'opacity-30'}">📧</span>
                    </td>
                    <td class="px-3 py-2 text-gray-500 text-xs col-created hidden">${formatDate(p.created_at)}</td>
                    <td class="px-3 py-2 text-center">
                        <button onclick="openPatientDetail(${p.id})" class="px-2 py-1 bg-gray-100 hover:bg-gray-200 rounded text-xs">📋 Detay</button>
                    </td>
                </tr>
            `;
        }).join('');
    }

    // İstatistikleri güncelle
    function updateStats() {
        document.getElementById('totalCount').textContent = state.total;
        // Active/blocked sayıları API'den gelebilir
        
        const selectedCount = state.selectedPatients.length;
        document.getElementById('selectedStats').classList.toggle('hidden', selectedCount === 0);
        document.getElementById('selectedCount').textContent = selectedCount;
        document.getElementById('bulkActions').classList.toggle('hidden', selectedCount === 0);
        
        // Showing text
        const start = (state.page - 1) * state.limit + 1;
        const end = Math.min(state.page * state.limit, state.total);
        document.getElementById('showingText').textContent = `${start}-${end} / ${state.total}`;
    }

    // Pagination render
    function renderPagination(totalPages) {
        const container = document.getElementById('pagination');
        if (totalPages <= 1) {
            container.innerHTML = '';
            return;
        }
        
        let html = '';
        if (state.page > 1) {
            html += `<button onclick="goToPage(${state.page - 1})" class="px-3 py-1 bg-gray-100 hover:bg-gray-200 rounded">◀</button>`;
        }
        
        for (let i = 1; i <= totalPages; i++) {
            if (i === 1 || i === totalPages || (i >= state.page - 2 && i <= state.page + 2)) {
                html += `<button onclick="goToPage(${i})" class="px-3 py-1 ${i === state.page ? 'bg-blue-600 text-white' : 'bg-gray-100 hover:bg-gray-200'} rounded">${i}</button>`;
            } else if (i === state.page - 3 || i === state.page + 3) {
                html += '<span class="px-2">...</span>';
            }
        }
        
        if (state.page < totalPages) {
            html += `<button onclick="goToPage(${state.page + 1})" class="px-3 py-1 bg-gray-100 hover:bg-gray-200 rounded">▶</button>`;
        }
        
        container.innerHTML = html;
    }

    function goToPage(page) {
        state.page = page;
        loadPatients();
    }

    // Seçim işlemleri
    function toggleSelect(id) {
        const idx = state.selectedPatients.indexOf(id);
        if (idx > -1) {
            state.selectedPatients.splice(idx, 1);
        } else {
            state.selectedPatients.push(id);
        }
        updateStats();
        renderPatients();
    }

    function toggleSelectAll() {
        const allChecked = document.getElementById('selectAll').checked;
        state.selectedPatients = allChecked ? state.patients.map(p => p.id) : [];
        updateStats();
        renderPatients();
    }

    // Sıralama
    function sortBy(field) {
        if (state.sort === field) {
            state.order = state.order === 'asc' ? 'desc' : 'asc';
        } else {
            state.sort = field;
            state.order = 'asc';
        }
        loadPatients();
    }

    // Sütun görünürlüğü
    function toggleColumnSettings() {
        document.getElementById('columnSettings').classList.toggle('hidden');
    }

    function toggleColumn(col) {
        const cells = document.querySelectorAll(`.col-${col}`);
        const checkbox = document.getElementById(`col${col.charAt(0).toUpperCase() + col.slice(1)}`);
        cells.forEach(cell => cell.classList.toggle('hidden', !checkbox.checked));
    }

    // Modal işlemleri
    function openModal(id) {
        document.getElementById(id).classList.add('active');
    }

    function closeModal(id) {
        document.getElementById(id).classList.remove('active');
    }

    // Yeni hasta ekle
    function openAddPatientModal() {
        state.currentPatient = null;
        document.getElementById('patientModalTitle').textContent = 'Yeni Hasta';
        document.getElementById('patientForm').reset();
        document.getElementById('patientId').value = '';
        document.getElementById('patientTags').innerHTML = '';
        document.getElementById('blockedWarning').classList.add('hidden');
        document.getElementById('blockBtn').classList.remove('hidden');
        document.getElementById('smsEnabled').checked = true;
        document.getElementById('whatsappEnabled').checked = true;
        document.getElementById('emailEnabled').checked = true;
        setPatientTab('info');
        openModal('patientModal');
    }

    // Hasta detayı
    async function openPatientDetail(id) {
        try {
            const response = await fetch(`${API_URL}/patients.php?id=${id}`);
            const data = await response.json();
            
            if (data.success) {
                state.currentPatient = data.patient;
                populatePatientForm(data.patient);
                openModal('patientModal');
            }
        } catch (error) {
            console.error('Detay yükleme hatası:', error);
        }
    }

    function populatePatientForm(patient) {
        document.getElementById('patientModalTitle').textContent = 'Hasta Detayı';
        document.getElementById('patientId').value = patient.id;
        document.getElementById('firstName').value = patient.first_name || '';
        document.getElementById('lastName').value = patient.last_name || '';
        document.getElementById('phone').value = patient.phone || '';
        document.getElementById('email').value = patient.email || '';
        document.getElementById('birthDate').value = patient.birth_date || '';
        document.getElementById('gender').value = patient.gender || '';
        document.getElementById('tcNo').value = patient.tc_no || '';
        document.getElementById('address').value = patient.address || '';
        document.getElementById('patientNotes').value = patient.notes || '';
        document.getElementById('smsEnabled').checked = patient.sms_enabled != 0;
        document.getElementById('whatsappEnabled').checked = patient.whatsapp_enabled != 0;
        document.getElementById('emailEnabled').checked = patient.email_enabled != 0;
        
        // Kara liste durumu
        if (patient.is_blocked == 1) {
            document.getElementById('blockedWarning').classList.remove('hidden');
            document.getElementById('blockReason').textContent = 'Sebep: ' + (patient.block_reason || 'Belirtilmemiş');
            document.getElementById('blockBtn').classList.add('hidden');
        } else {
            document.getElementById('blockedWarning').classList.add('hidden');
            document.getElementById('blockBtn').classList.remove('hidden');
        }
        
        // Etiketler
        renderPatientTags(patient.tags || []);
        
        // Notlar
        renderNotes(patient.notes_list || patient.notes || []);
        
        // Randevular
        renderAppointments(patient.appointments || []);
        
        // İletişim
        renderCommunications(patient.communications || []);
        
        setPatientTab('info');
    }

    function renderPatientTags(patientTags) {
        const container = document.getElementById('patientTags');
        container.innerHTML = patientTags.map(t => `
            <span class="tag-badge" style="background:${t.color}20;color:${t.color}">
                ${t.name}
                <button type="button" onclick="removePatientTag(${t.id})" class="ml-1 hover:opacity-70">×</button>
            </span>
        `).join('');
        
        // Tag butonlarını güncelle
        document.querySelectorAll('.tag-btn').forEach(btn => {
            const tagId = parseInt(btn.dataset.tagId);
            const hasTag = patientTags.some(t => t.id === tagId);
            btn.classList.toggle('hidden', hasTag);
        });
    }

    function togglePatientTag(tagId, tagName, tagColor) {
        if (!state.currentPatient) state.currentPatient = { tags: [] };
        if (!state.currentPatient.tags) state.currentPatient.tags = [];
        
        const exists = state.currentPatient.tags.find(t => t.id === tagId);
        if (!exists) {
            state.currentPatient.tags.push({ id: tagId, name: tagName, color: tagColor });
        }
        renderPatientTags(state.currentPatient.tags);
    }

    function removePatientTag(tagId) {
        if (!state.currentPatient || !state.currentPatient.tags) return;
        state.currentPatient.tags = state.currentPatient.tags.filter(t => t.id !== tagId);
        renderPatientTags(state.currentPatient.tags);
    }

    // Sekmeler
    function setPatientTab(tab) {
        state.currentPatientTab = tab;
        ['info', 'notes', 'appointments', 'communication'].forEach(t => {
            document.getElementById(`tab${t.charAt(0).toUpperCase() + t.slice(1)}`).className = t === tab ? 'px-4 py-2 font-medium border-b-2 border-blue-600 text-blue-600' : 'px-4 py-2 font-medium text-gray-500';
            document.getElementById(`tabContent${t.charAt(0).toUpperCase() + t.slice(1)}`).classList.toggle('hidden', t !== tab);
        });
    }

    // Notlar
    function setNoteType(type) {
        state.currentNoteType = type;
        document.querySelectorAll('.note-type-btn').forEach(btn => {
            btn.className = btn.dataset.type === type ? 'note-type-btn px-3 py-1.5 bg-blue-100 text-blue-700 rounded text-sm' : 'note-type-btn px-3 py-1.5 bg-gray-100 text-gray-700 rounded text-sm';
        });
    }

    async function addNote() {
        const text = document.getElementById('newNoteText').value.trim();
        if (!text || !state.currentPatient) return;
        
        try {
            const response = await fetch(`${API_URL}/patient-notes.php`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    patient_id: state.currentPatient.id,
                    note: text,
                    note_type: state.currentNoteType
                })
            });
            
            const data = await response.json();
            if (data.success) {
                document.getElementById('newNoteText').value = '';
                openPatientDetail(state.currentPatient.id);
                setPatientTab('notes');
            }
        } catch (error) {
            console.error('Not ekleme hatası:', error);
        }
    }

    function renderNotes(notes) {
        const container = document.getElementById('notesList');
        if (!notes || notes.length === 0) {
            container.innerHTML = '<div class="text-center py-8 text-gray-500">Henüz not eklenmemiş</div>';
            return;
        }
        
        const typeIcons = { general: '📝', medical: '🏥', billing: '💰', complaint: '⚠️', allergy: '🚨' };
        const typeColors = { general: '#3b82f6', medical: '#ef4444', billing: '#22c55e', complaint: '#f59e0b', allergy: '#dc2626' };
        
        container.innerHTML = notes.map(n => `
            <div class="p-3 bg-gray-50 rounded-lg border-l-3" style="border-left-color: ${typeColors[n.note_type] || '#3b82f6'}">
                <div class="flex justify-between items-center mb-1">
                    <span class="text-xs font-medium" style="color: ${typeColors[n.note_type] || '#3b82f6'}">${typeIcons[n.note_type] || '📝'} ${n.note_type}</span>
                    <span class="text-xs text-gray-400">${formatDate(n.created_at)} • ${n.created_by_name || 'Sistem'}</span>
                </div>
                <div class="text-sm">${n.note}</div>
            </div>
        `).join('');
    }

    function renderAppointments(appointments) {
        const container = document.getElementById('appointmentsList');
        if (!appointments || appointments.length === 0) {
            container.innerHTML = '<div class="text-center py-8 text-gray-500">Randevu kaydı yok</div>';
            return;
        }
        
        const statusColors = { pending: '#94a3b8', confirmed: '#3b82f6', completed: '#22c55e', cancelled: '#6b7280', no_show: '#ef4444' };
        const statusNames = { pending: 'Beklemede', confirmed: 'Onaylı', completed: 'Geldi', cancelled: 'İptal', no_show: 'Gelmedi' };
        
        container.innerHTML = appointments.map(a => `
            <div class="flex items-center gap-3 p-3 border-b">
                <div class="w-3 h-3 rounded-full" style="background: ${statusColors[a.status] || '#94a3b8'}"></div>
                <div class="flex-1">
                    <div class="font-medium text-sm">${formatDate(a.appointment_date)} • ${a.start_time?.slice(0,5)}</div>
                    <div class="text-xs text-gray-500">${a.service_name} • ${a.staff_name}</div>
                </div>
                <span class="px-2 py-0.5 rounded text-xs" style="background: ${statusColors[a.status]}20; color: ${statusColors[a.status]}">${statusNames[a.status] || a.status}</span>
            </div>
        `).join('');
    }

    function renderCommunications(communications) {
        const container = document.getElementById('communicationList');
        if (!communications || communications.length === 0) {
            container.innerHTML = '<div class="text-center py-8 text-gray-500">İletişim kaydı yok</div>';
            return;
        }
        
        const typeIcons = { sms: '📱', whatsapp: '💬', email: '📧', call: '📞' };
        
        container.innerHTML = communications.map(c => `
            <div class="flex items-start gap-3 p-3 border-b">
                <span class="text-lg">${typeIcons[c.type] || '📝'}</span>
                <div class="flex-1">
                    <div class="flex items-center gap-2 mb-1">
                        <span class="text-xs font-medium ${c.direction === 'incoming' ? 'text-green-600' : 'text-blue-600'}">${c.direction === 'incoming' ? '← Gelen' : '→ Giden'}</span>
                        <span class="text-xs text-gray-400">${formatDateTime(c.created_at)}</span>
                    </div>
                    <div class="text-sm">${c.message || c.subject || '-'}</div>
                </div>
                <span class="text-xs px-2 py-0.5 rounded ${c.status === 'delivered' ? 'bg-green-100 text-green-700' : c.status === 'read' ? 'bg-blue-100 text-blue-700' : 'bg-amber-100 text-amber-700'}">${c.status}</span>
            </div>
        `).join('');
    }

    // Hasta kaydet
    async function savePatient(e) {
        e.preventDefault();
        
        const formData = {
            id: document.getElementById('patientId').value || null,
            first_name: document.getElementById('firstName').value,
            last_name: document.getElementById('lastName').value,
            phone: document.getElementById('phone').value,
            email: document.getElementById('email').value,
            birth_date: document.getElementById('birthDate').value,
            gender: document.getElementById('gender').value,
            tc_no: document.getElementById('tcNo').value,
            address: document.getElementById('address').value,
            notes: document.getElementById('patientNotes').value,
            sms_enabled: document.getElementById('smsEnabled').checked ? 1 : 0,
            whatsapp_enabled: document.getElementById('whatsappEnabled').checked ? 1 : 0,
            email_enabled: document.getElementById('emailEnabled').checked ? 1 : 0,
            tags: state.currentPatient?.tags?.map(t => t.id) || []
        };
        
        try {
            const method = formData.id ? 'PUT' : 'POST';
            const response = await fetch(`${API_URL}/patients.php`, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();
            if (data.success) {
                closeModal('patientModal');
                loadPatients();
            } else {
                alert(data.error || 'Hata oluştu');
            }
        } catch (error) {
            console.error('Kayıt hatası:', error);
        }
    }

    // Kara liste
    async function blockPatient() {
        const reason = prompt('Kara liste sebebini girin:');
        if (!reason) return;
        
        try {
            const response = await fetch(`${API_URL}/patients.php`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    id: state.currentPatient.id,
                    is_blocked: true,
                    block_reason: reason
                })
            });
            
            const data = await response.json();
            if (data.success) {
                closeModal('patientModal');
                loadPatients();
            }
        } catch (error) {
            console.error('Kara liste hatası:', error);
        }
    }

    async function unblockPatient() {
        try {
            const response = await fetch(`${API_URL}/patients.php`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    id: state.currentPatient.id,
                    is_blocked: false
                })
            });
            
            const data = await response.json();
            if (data.success) {
                closeModal('patientModal');
                loadPatients();
            }
        } catch (error) {
            console.error('Kara listeden çıkarma hatası:', error);
        }
    }

    // Toplu işlemler
    async function bulkBlock() {
        if (!confirm(`${state.selectedPatients.length} hastayı kara listeye almak istediğinize emin misiniz?`)) return;
        // Bulk API çağrısı yapılacak
        alert('Toplu kara listeye alma işlemi yapıldı');
        state.selectedPatients = [];
        loadPatients();
    }

    // Excel export
    function exportPatients() {
        const url = `${API_URL}/patients.php?export=excel&status=${document.getElementById('statusFilter').value}&search=${state.search}`;
        window.open(url, '_blank');
    }

    // Import
    function openImportModal() {
        document.getElementById('importFile').value = '';
        document.getElementById('importPreview').classList.add('hidden');
        openModal('importModal');
    }

    function previewImport(input) {
        const file = input.files[0];
        if (!file) return;
        
        const reader = new FileReader();
        reader.onload = function(e) {
            const text = e.target.result;
            const lines = text.split('\n').filter(l => l.trim());
            const headers = lines[0].split(/[,\t;]/).map(h => h.trim().replace(/"/g, ''));
            const rows = lines.slice(1, 6).map(line => line.split(/[,\t;]/).map(v => v.trim().replace(/"/g, '')));
            
            state.importData = { headers, totalRows: lines.length - 1, file };
            
            document.getElementById('importCount').textContent = lines.length - 1;
            document.getElementById('importHeaders').innerHTML = headers.slice(0, 6).map(h => `<th class="px-2 py-1 text-left bg-gray-100">${h}</th>`).join('');
            document.getElementById('importBody').innerHTML = rows.map(row => `<tr>${row.slice(0, 6).map(v => `<td class="px-2 py-1 border-t">${v}</td>`).join('')}</tr>`).join('');
            document.getElementById('importPreview').classList.remove('hidden');
        };
        reader.readAsText(file, 'UTF-8');
    }

    async function executeImport() {
        if (!state.importData) return;
        
        const formData = new FormData();
        formData.append('file', state.importData.file);
        
        try {
            const response = await fetch(`${API_URL}/patients-import.php`, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            if (data.success) {
                alert(`${data.imported} hasta başarıyla içe aktarıldı!`);
                closeModal('importModal');
                loadPatients();
            } else {
                alert(data.error || 'İçe aktarma hatası');
            }
        } catch (error) {
            console.error('İçe aktarma hatası:', error);
        }
    }

    // Yardımcı fonksiyonlar
    function getTagColor(tagName) {
        const tag = tags.find(t => t.name === tagName);
        return tag ? tag.color : '#6B7280';
    }

    function calculateAge(birthDate) {
        const today = new Date();
        const birth = new Date(birthDate);
        let age = today.getFullYear() - birth.getFullYear();
        if (today.getMonth() < birth.getMonth() || (today.getMonth() === birth.getMonth() && today.getDate() < birth.getDate())) age--;
        return age;
    }

    function formatDate(dateStr) {
        if (!dateStr) return '-';
        const date = new Date(dateStr);
        return date.toLocaleDateString('tr-TR', { day: 'numeric', month: 'short', year: 'numeric' });
    }

    function formatDateTime(dateStr) {
        if (!dateStr) return '-';
        const date = new Date(dateStr);
        return date.toLocaleDateString('tr-TR', { day: 'numeric', month: 'short', hour: '2-digit', minute: '2-digit' });
    }

    // Click outside to close
    document.addEventListener('click', function(e) {
        if (!e.target.closest('#columnSettings') && !e.target.closest('[onclick="toggleColumnSettings()"]')) {
            document.getElementById('columnSettings').classList.add('hidden');
        }
    });
    </script>
</body>
</html>
