<?php
/**
 * Randevu Sistemi V7 - Admin Ana Sayfa (Takvim)
 * o2nar.com/test/admin/index.php
 */

require_once __DIR__ . '/../includes/auth.php';
requireAdmin();

$currentUser = currentUser();
$selectedDate = $_GET['date'] ?? date('Y-m-d');

// Personel listesi
$staffStmt = db()->query("SELECT id, first_name, last_name, color, role FROM staff WHERE is_active = 1 ORDER BY sort_order, first_name");
$staffList = $staffStmt->fetchAll();

// Hizmet listesi
$servicesStmt = db()->query("SELECT id, name, color, duration FROM services WHERE is_active = 1 ORDER BY sort_order, name");
$servicesList = $servicesStmt->fetchAll();

// Randevu durumları
$statuses = [
    'pending' => ['name' => 'Beklemede', 'color' => '#94a3b8', 'bg' => '#f1f5f9', 'icon' => '📅'],
    'confirmed' => ['name' => 'Onaylı', 'color' => '#3b82f6', 'bg' => '#dbeafe', 'icon' => '✓'],
    'completed' => ['name' => 'Geldi', 'color' => '#22c55e', 'bg' => '#dcfce7', 'icon' => '✅'],
    'cancelled' => ['name' => 'İptal', 'color' => '#6b7280', 'bg' => '#e5e7eb', 'icon' => '🚫'],
    'no_show' => ['name' => 'Gelmedi', 'color' => '#ef4444', 'bg' => '#fee2e2', 'icon' => '❌'],
];

$pageTitle = 'Randevu Takvimi';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?> - <?= CLINIC_NAME ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .slot-height { height: 40px; }
        .appointment-card { 
            transition: all 0.2s ease;
            cursor: grab;
        }
        .appointment-card:active { cursor: grabbing; }
        .appointment-card:hover { 
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateY(-1px);
        }
        .resize-handle {
            position: absolute;
            left: 0;
            right: 0;
            height: 6px;
            cursor: ns-resize;
        }
        .resize-handle-top { top: 0; }
        .resize-handle-bottom { bottom: 0; }
        .time-slot:hover { background: #f8fafc; }
        .dragging { opacity: 0.8; z-index: 100; }
        .drop-target { background: #dbeafe !important; }
        .tooltip {
            position: absolute;
            left: 100%;
            top: 0;
            margin-left: 8px;
            background: #1f2937;
            color: white;
            padding: 12px 16px;
            border-radius: 8px;
            font-size: 12px;
            max-width: 280px;
            z-index: 9999;
            box-shadow: 0 4px 20px rgba(0,0,0,0.25);
            display: none;
        }
        .appointment-card:hover .tooltip { display: block; }
        .modal { display: none; }
        .modal.active { display: flex; }
        .blocked-slot {
            background: repeating-linear-gradient(45deg, #f3f4f6, #f3f4f6 10px, white 10px, white 20px);
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    <!-- Header -->
    <header class="bg-white border-b border-gray-200 sticky top-0 z-50">
        <div class="flex items-center justify-between px-4 py-2">
            <div class="flex items-center gap-4">
                <h1 class="text-lg font-bold text-blue-800">📅 Randevu Takvimi</h1>
                
                <!-- Tarih Navigasyonu -->
                <div class="flex items-center gap-1">
                    <button onclick="navigateDate(-1)" class="px-3 py-1.5 bg-gray-100 hover:bg-gray-200 rounded text-sm">◀</button>
                    <button onclick="goToToday()" class="px-3 py-1.5 bg-amber-100 hover:bg-amber-200 text-amber-800 rounded text-sm font-medium">Bugün</button>
                    <input type="date" id="selectedDate" value="<?= e($selectedDate) ?>" onchange="changeDate(this.value)" class="px-2 py-1.5 border rounded text-sm">
                    <button onclick="navigateDate(1)" class="px-3 py-1.5 bg-gray-100 hover:bg-gray-200 rounded text-sm">▶</button>
                </div>
                
                <span class="text-sm font-medium text-gray-700" id="dateDisplay"></span>
                
                <!-- Görünüm Modu -->
                <div class="flex bg-gray-100 rounded p-0.5">
                    <button onclick="setViewMode('single')" id="btnSingle" class="px-3 py-1 text-xs rounded">Takvimim</button>
                    <button onclick="setViewMode('all')" id="btnAll" class="px-3 py-1 text-xs rounded">Kuşbakışı</button>
                </div>
                
                <!-- Personel Seçimi (Tek kişi modunda) -->
                <select id="staffSelect" onchange="changeStaff(this.value)" class="px-2 py-1.5 border rounded text-sm">
                    <?php foreach ($staffList as $staff): ?>
                    <option value="<?= $staff['id'] ?>" <?= $staff['id'] == $currentUser['id'] ? 'selected' : '' ?>>
                        <?= e($staff['first_name'] . ' ' . $staff['last_name']) ?>
                    </option>
                    <?php endforeach; ?>
                </select>
                
                <!-- Arama -->
                <input type="text" id="searchInput" placeholder="🔍 Hasta ara..." onkeyup="filterAppointments()" class="px-3 py-1.5 border rounded text-sm w-40">
            </div>
            
            <div class="flex items-center gap-2">
                <a href="patients.php" class="px-3 py-1.5 bg-indigo-100 text-indigo-700 hover:bg-indigo-200 rounded text-sm font-medium">👥 Hastalar</a>
                <a href="services.php" class="px-3 py-1.5 bg-gray-100 hover:bg-gray-200 rounded text-sm">🏷️ Hizmetler</a>
                <a href="staff.php" class="px-3 py-1.5 bg-gray-100 hover:bg-gray-200 rounded text-sm">🩺 Personel</a>
                <button onclick="openBlockModal()" class="px-3 py-1.5 bg-amber-100 text-amber-800 hover:bg-amber-200 rounded text-sm">🚫 Bloke</button>
                <button onclick="openAddModal()" class="px-3 py-1.5 bg-blue-600 text-white hover:bg-blue-700 rounded text-sm font-medium">➕ Randevu</button>
                <div class="w-px h-6 bg-gray-300 mx-2"></div>
                <span class="text-sm text-gray-600"><?= e($currentUser['name']) ?></span>
                <a href="logout.php" class="text-sm text-red-600 hover:text-red-800">Çıkış</a>
            </div>
        </div>
        
        <!-- Hizmet Renkleri -->
        <div class="flex items-center gap-4 px-4 py-1.5 bg-gray-50 border-t text-xs">
            <?php foreach ($servicesList as $service): ?>
            <div class="flex items-center gap-1">
                <div class="w-3 h-3 rounded" style="background: <?= e($service['color']) ?>20; border-left: 2px solid <?= e($service['color']) ?>"></div>
                <span class="text-gray-600"><?= e($service['name']) ?></span>
            </div>
            <?php endforeach; ?>
            <span class="text-gray-300">|</span>
            <?php foreach ($statuses as $key => $status): ?>
            <div class="flex items-center gap-1">
                <span><?= $status['icon'] ?></span>
                <span class="text-gray-600"><?= $status['name'] ?></span>
            </div>
            <?php endforeach; ?>
        </div>
    </header>

    <!-- Takvim Grid -->
    <div class="p-4 overflow-auto" style="height: calc(100vh - 120px);">
        <div id="calendarContainer" class="flex min-w-fit">
            <!-- Saat Sütunu -->
            <div class="w-14 flex-shrink-0">
                <div class="h-11 border-b border-gray-200"></div>
                <?php 
                $startHour = (int)substr(WORKING_HOURS_START, 0, 2);
                $endHour = (int)substr(WORKING_HOURS_END, 0, 2);
                for ($h = $startHour; $h < $endHour; $h++):
                    for ($m = 0; $m < 60; $m += SLOT_DURATION):
                        $time = sprintf('%02d:%02d', $h, $m);
                ?>
                <div class="slot-height flex items-start justify-end pr-2 text-xs text-gray-500 border-b <?= $m == 0 ? 'font-semibold border-gray-200' : 'border-gray-100' ?>">
                    <?= $time ?>
                </div>
                <?php endfor; endfor; ?>
            </div>
            
            <!-- Personel Sütunları -->
            <div id="staffColumns" class="flex flex-1">
                <!-- JavaScript ile doldurulacak -->
            </div>
        </div>
    </div>

    <!-- Randevu Ekleme/Düzenleme Modal -->
    <div id="appointmentModal" class="modal fixed inset-0 bg-black/50 items-center justify-center z-50">
        <div class="bg-white rounded-xl p-6 w-full max-w-lg max-h-[90vh] overflow-auto shadow-2xl">
            <div class="flex justify-between items-center mb-4">
                <h2 id="modalTitle" class="text-lg font-semibold text-gray-800">Yeni Randevu</h2>
                <button onclick="closeModal('appointmentModal')" class="text-gray-400 hover:text-gray-600 text-2xl">&times;</button>
            </div>
            
            <form id="appointmentForm" onsubmit="saveAppointment(event)">
                <input type="hidden" id="appointmentId" name="id">
                
                <!-- Hasta Seçimi -->
                <div class="mb-4">
                    <div class="flex gap-2 mb-2">
                        <button type="button" onclick="setPatientMode('existing')" id="btnExisting" class="flex-1 py-2 rounded bg-blue-600 text-white text-sm font-medium">Mevcut Hasta</button>
                        <button type="button" onclick="setPatientMode('new')" id="btnNew" class="flex-1 py-2 rounded bg-gray-100 text-gray-700 text-sm font-medium">Yeni Hasta</button>
                    </div>
                    
                    <div id="existingPatientSection">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Hasta Seç</label>
                        <select id="patientId" name="patient_id" class="w-full px-3 py-2 border rounded-lg" required>
                            <option value="">Hasta arayın...</option>
                        </select>
                        <input type="text" id="patientSearch" placeholder="🔍 İsim veya telefon ile ara..." onkeyup="searchPatients(this.value)" class="w-full px-3 py-2 border rounded-lg mt-2">
                        <div id="patientResults" class="max-h-40 overflow-auto border rounded-lg mt-1 hidden"></div>
                    </div>
                    
                    <div id="newPatientSection" class="hidden bg-gray-50 p-4 rounded-lg">
                        <div class="grid grid-cols-2 gap-3">
                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-1">Ad *</label>
                                <input type="text" id="newFirstName" class="w-full px-3 py-2 border rounded text-sm">
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-1">Soyad *</label>
                                <input type="text" id="newLastName" class="w-full px-3 py-2 border rounded text-sm">
                            </div>
                        </div>
                        <div class="mt-3">
                            <label class="block text-xs font-medium text-gray-700 mb-1">Telefon *</label>
                            <input type="tel" id="newPhone" class="w-full px-3 py-2 border rounded text-sm" placeholder="+90 5XX XXX XX XX">
                        </div>
                        <div id="phoneWarning" class="mt-2 hidden bg-amber-50 border border-amber-200 rounded-lg p-3"></div>
                    </div>
                </div>
                
                <!-- Personel -->
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Personel</label>
                    <select id="staffId" name="staff_id" class="w-full px-3 py-2 border rounded-lg" required>
                        <?php foreach ($staffList as $staff): ?>
                        <option value="<?= $staff['id'] ?>"><?= e($staff['first_name'] . ' ' . $staff['last_name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Hizmet -->
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Hizmet</label>
                    <select id="serviceId" name="service_id" onchange="updateDuration()" class="w-full px-3 py-2 border rounded-lg" required>
                        <option value="">Seçiniz...</option>
                        <?php foreach ($servicesList as $service): ?>
                        <option value="<?= $service['id'] ?>" data-duration="<?= $service['duration'] ?>"><?= e($service['name']) ?> (<?= $service['duration'] ?> dk)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Tarih & Saat -->
                <div class="grid grid-cols-3 gap-3 mb-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Tarih</label>
                        <input type="date" id="appointmentDate" name="appointment_date" class="w-full px-3 py-2 border rounded-lg" required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Başlangıç</label>
                        <input type="time" id="startTime" name="start_time" step="900" class="w-full px-3 py-2 border rounded-lg" required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Bitiş</label>
                        <input type="time" id="endTime" name="end_time" step="900" class="w-full px-3 py-2 border rounded-lg" required>
                    </div>
                </div>
                
                <!-- Durum -->
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Durum</label>
                    <select id="status" name="status" class="w-full px-3 py-2 border rounded-lg">
                        <?php foreach ($statuses as $key => $status): ?>
                        <option value="<?= $key ?>"><?= $status['icon'] ?> <?= $status['name'] ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Notlar -->
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Notlar</label>
                    <textarea id="notes" name="notes" rows="2" class="w-full px-3 py-2 border rounded-lg" placeholder="Randevu notları..."></textarea>
                </div>
                
                <!-- Butonlar -->
                <div class="flex gap-2 justify-between">
                    <button type="button" id="deleteBtn" onclick="deleteAppointment()" class="px-4 py-2 bg-red-100 text-red-700 hover:bg-red-200 rounded-lg hidden">🗑️ Sil</button>
                    <div class="flex gap-2 ml-auto">
                        <button type="button" onclick="closeModal('appointmentModal')" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg">İptal</button>
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white hover:bg-blue-700 rounded-lg font-medium">💾 Kaydet</button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Bloke Modal -->
    <div id="blockModal" class="modal fixed inset-0 bg-black/50 items-center justify-center z-50">
        <div class="bg-white rounded-xl p-6 w-full max-w-md shadow-2xl">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-lg font-semibold text-gray-800">🚫 Zaman Bloke Et</h2>
                <button onclick="closeModal('blockModal')" class="text-gray-400 hover:text-gray-600 text-2xl">&times;</button>
            </div>
            
            <form id="blockForm" onsubmit="saveBlock(event)">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Personel</label>
                    <select id="blockStaffId" name="staff_id" class="w-full px-3 py-2 border rounded-lg" required>
                        <?php foreach ($staffList as $staff): ?>
                        <option value="<?= $staff['id'] ?>"><?= e($staff['first_name'] . ' ' . $staff['last_name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Tür</label>
                    <select id="blockType" name="block_type" class="w-full px-3 py-2 border rounded-lg">
                        <option value="break">☕ Mola</option>
                        <option value="leave">🏖️ İzin</option>
                        <option value="meeting">👥 Toplantı</option>
                        <option value="other">📌 Diğer</option>
                    </select>
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Tarih</label>
                    <input type="date" id="blockDate" name="block_date" class="w-full px-3 py-2 border rounded-lg" required>
                </div>
                
                <div class="grid grid-cols-2 gap-3 mb-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Başlangıç</label>
                        <input type="time" id="blockStartTime" name="start_time" step="900" class="w-full px-3 py-2 border rounded-lg" required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Bitiş</label>
                        <input type="time" id="blockEndTime" name="end_time" step="900" class="w-full px-3 py-2 border rounded-lg" required>
                    </div>
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Açıklama</label>
                    <input type="text" id="blockReason" name="reason" class="w-full px-3 py-2 border rounded-lg" placeholder="Opsiyonel...">
                </div>
                
                <div class="flex gap-2 justify-end">
                    <button type="button" onclick="closeModal('blockModal')" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-lg">İptal</button>
                    <button type="submit" class="px-4 py-2 bg-amber-500 text-white hover:bg-amber-600 rounded-lg font-medium">🚫 Bloke Et</button>
                </div>
            </form>
        </div>
    </div>

    <script>
    // Konfigürasyon
    const CONFIG = {
        slotDuration: <?= SLOT_DURATION ?>,
        slotHeight: 40,
        startHour: <?= (int)substr(WORKING_HOURS_START, 0, 2) ?>,
        endHour: <?= (int)substr(WORKING_HOURS_END, 0, 2) ?>,
        apiUrl: '<?= API_URL ?>',
        currentUserId: <?= $currentUser['id'] ?>
    };

    // State
    let state = {
        date: '<?= e($selectedDate) ?>',
        viewMode: 'single',
        selectedStaff: <?= $currentUser['id'] ?>,
        appointments: [],
        blocks: [],
        patients: [],
        patientMode: 'existing',
        dragState: null,
        resizeState: null
    };

    // Personel ve hizmet verileri
    const staffList = <?= json_encode($staffList) ?>;
    const servicesList = <?= json_encode($servicesList) ?>;
    const statuses = <?= json_encode($statuses) ?>;

    // Sayfa yüklendiğinde
    document.addEventListener('DOMContentLoaded', function() {
        updateDateDisplay();
        setViewMode('single');
        loadAppointments();
    });

    // Tarih gösterimi güncelle
    function updateDateDisplay() {
        const date = new Date(state.date);
        const options = { weekday: 'long', day: 'numeric', month: 'long', year: 'numeric' };
        document.getElementById('dateDisplay').textContent = date.toLocaleDateString('tr-TR', options);
    }

    // Tarih navigasyonu
    function navigateDate(direction) {
        const date = new Date(state.date);
        date.setDate(date.getDate() + direction);
        state.date = date.toISOString().split('T')[0];
        document.getElementById('selectedDate').value = state.date;
        updateDateDisplay();
        loadAppointments();
    }

    function changeDate(newDate) {
        state.date = newDate;
        updateDateDisplay();
        loadAppointments();
    }

    function goToToday() {
        state.date = new Date().toISOString().split('T')[0];
        document.getElementById('selectedDate').value = state.date;
        updateDateDisplay();
        loadAppointments();
    }

    // Görünüm modu
    function setViewMode(mode) {
        state.viewMode = mode;
        document.getElementById('btnSingle').className = mode === 'single' ? 'px-3 py-1 text-xs rounded bg-white shadow font-medium text-blue-700' : 'px-3 py-1 text-xs rounded text-gray-600';
        document.getElementById('btnAll').className = mode === 'all' ? 'px-3 py-1 text-xs rounded bg-white shadow font-medium text-blue-700' : 'px-3 py-1 text-xs rounded text-gray-600';
        document.getElementById('staffSelect').style.display = mode === 'single' ? 'block' : 'none';
        renderCalendar();
    }

    function changeStaff(staffId) {
        state.selectedStaff = parseInt(staffId);
        renderCalendar();
    }

    // Randevuları yükle
    async function loadAppointments() {
        try {
            const response = await fetch(`${CONFIG.apiUrl}/appointments.php?date=${state.date}`);
            const data = await response.json();
            if (data.success) {
                state.appointments = data.appointments;
                state.blocks = data.blocks || [];
                renderCalendar();
            }
        } catch (error) {
            console.error('Randevular yüklenemedi:', error);
        }
    }

    // Takvimi render et
    function renderCalendar() {
        const container = document.getElementById('staffColumns');
        container.innerHTML = '';
        
        const visibleStaff = state.viewMode === 'single' 
            ? staffList.filter(s => s.id == state.selectedStaff)
            : staffList;
        
        const columnWidth = state.viewMode === 'single' ? 350 : Math.max(160, (window.innerWidth - 100) / visibleStaff.length);
        
        visibleStaff.forEach(staff => {
            const staffAppts = state.appointments.filter(a => a.staff_id == staff.id);
            const staffBlocks = state.blocks.filter(b => b.staff_id == staff.id);
            const positioned = calculatePositions(staffAppts);
            
            const column = document.createElement('div');
            column.className = 'border-l border-gray-200 flex-shrink-0';
            column.style.width = columnWidth + 'px';
            
            // Başlık
            column.innerHTML = `
                <div class="h-11 border-b border-gray-200 flex items-center justify-center font-semibold text-sm" 
                     style="background: linear-gradient(to bottom, ${staff.color}20, transparent); border-top: 3px solid ${staff.color}">
                    ${staff.first_name} ${staff.last_name}
                </div>
                <div class="relative" id="grid-${staff.id}"></div>
            `;
            
            container.appendChild(column);
            
            const grid = document.getElementById(`grid-${staff.id}`);
            
            // Zaman slotları
            for (let h = CONFIG.startHour; h < CONFIG.endHour; h++) {
                for (let m = 0; m < 60; m += CONFIG.slotDuration) {
                    const time = `${String(h).padStart(2, '0')}:${String(m).padStart(2, '0')}`;
                    const slot = document.createElement('div');
                    slot.className = `slot-height border-b time-slot cursor-pointer ${m === 0 ? 'border-gray-200' : 'border-gray-100'}`;
                    slot.dataset.time = time;
                    slot.dataset.staffId = staff.id;
                    slot.onclick = () => openAddModal(staff.id, time);
                    grid.appendChild(slot);
                }
            }
            
            // Blokları render et
            staffBlocks.forEach(block => {
                const blockEl = createBlockElement(block, columnWidth);
                grid.appendChild(blockEl);
            });
            
            // Randevuları render et
            positioned.forEach(appt => {
                const apptEl = createAppointmentElement(appt, columnWidth);
                grid.appendChild(apptEl);
            });
        });
    }

    // Overlapping pozisyonları hesapla
    function calculatePositions(appointments) {
        const sorted = [...appointments].sort((a, b) => timeToMinutes(a.start_time) - timeToMinutes(b.start_time));
        const positioned = [];
        
        sorted.forEach(appt => {
            const apptStart = timeToMinutes(appt.start_time);
            const apptEnd = timeToMinutes(appt.end_time);
            
            const overlapping = positioned.filter(p => {
                const pStart = timeToMinutes(p.start_time);
                const pEnd = timeToMinutes(p.end_time);
                return !(apptEnd <= pStart || apptStart >= pEnd);
            });
            
            const usedColumns = overlapping.map(p => p.column);
            let column = 0;
            while (usedColumns.includes(column)) column++;
            
            const maxColumn = Math.max(column, ...overlapping.map(p => (p.maxColumns || 1) - 1));
            positioned.push({ ...appt, column, maxColumns: maxColumn + 1 });
            overlapping.forEach(p => p.maxColumns = Math.max(p.maxColumns || 1, maxColumn + 1));
        });
        
        // İkinci geçiş - maxColumns'u eşitle
        positioned.forEach(appt => {
            const overlapping = positioned.filter(p => {
                if (p.id === appt.id) return false;
                const apptStart = timeToMinutes(appt.start_time);
                const apptEnd = timeToMinutes(appt.end_time);
                const pStart = timeToMinutes(p.start_time);
                const pEnd = timeToMinutes(p.end_time);
                return !(apptEnd <= pStart || apptStart >= pEnd);
            });
            const maxCols = Math.max(appt.maxColumns, ...overlapping.map(p => p.maxColumns));
            appt.maxColumns = maxCols;
            overlapping.forEach(p => p.maxColumns = maxCols);
        });
        
        return positioned;
    }

    function timeToMinutes(time) {
        const [h, m] = time.split(':').map(Number);
        return h * 60 + m;
    }

    function minutesToTime(mins) {
        return `${String(Math.floor(mins / 60)).padStart(2, '0')}:${String(mins % 60).padStart(2, '0')}`;
    }

    // Randevu elementi oluştur
    function createAppointmentElement(appt, columnWidth) {
        const service = servicesList.find(s => s.id == appt.service_id) || { color: '#94a3b8', name: 'Bilinmeyen' };
        const status = statuses[appt.status] || statuses.pending;
        
        const startMins = timeToMinutes(appt.start_time) - (CONFIG.startHour * 60);
        const endMins = timeToMinutes(appt.end_time) - (CONFIG.startHour * 60);
        const top = (startMins / CONFIG.slotDuration) * CONFIG.slotHeight;
        const height = ((endMins - startMins) / CONFIG.slotDuration) * CONFIG.slotHeight - 2;
        
        const width = (columnWidth - 12) / (appt.maxColumns || 1);
        const left = 4 + (appt.column || 0) * width;
        
        const el = document.createElement('div');
        el.className = 'appointment-card absolute rounded text-xs overflow-hidden';
        el.style.cssText = `
            top: ${top}px;
            left: ${left}px;
            width: ${width - 4}px;
            height: ${height}px;
            background: ${appt.status === 'cancelled' ? '#f3f4f6' : service.color + '30'};
            border-left: 3px solid ${appt.status === 'cancelled' ? '#9ca3af' : service.color};
            opacity: ${appt.status === 'cancelled' ? '0.6' : '1'};
        `;
        el.dataset.id = appt.id;
        
        el.innerHTML = `
            <div class="resize-handle resize-handle-top"></div>
            <div class="p-1 pointer-events-none">
                <div class="flex justify-between items-center">
                    <span class="font-semibold">${appt.start_time.slice(0,5)}-${appt.end_time.slice(0,5)}</span>
                    <span class="text-[9px] px-1 rounded" style="background:${status.bg};color:${status.color}">${status.icon}</span>
                </div>
                <div class="truncate">${appt.patient_name || 'Hasta'}</div>
                <div class="text-gray-500 text-[10px]">${service.name}</div>
            </div>
            <div class="resize-handle resize-handle-bottom"></div>
            <div class="tooltip">
                <div class="font-semibold mb-1">${appt.patient_name || 'Hasta'}</div>
                <div class="text-gray-400 mb-2">${appt.patient_phone || ''}</div>
                ${appt.notes ? `<div>📝 ${appt.notes}</div>` : ''}
                ${appt.patient_notes ? `<div class="text-amber-400 mt-1">⚠️ ${appt.patient_notes}</div>` : ''}
            </div>
        `;
        
        // Çift tıklama - düzenleme
        el.ondblclick = (e) => {
            e.stopPropagation();
            openEditModal(appt);
        };
        
        // Drag başlat
        el.onmousedown = (e) => {
            if (e.target.classList.contains('resize-handle')) {
                startResize(e, appt, e.target.classList.contains('resize-handle-top') ? 'top' : 'bottom');
            } else {
                startDrag(e, appt);
            }
        };
        
        return el;
    }

    // Blok elementi oluştur
    function createBlockElement(block, columnWidth) {
        const startMins = timeToMinutes(block.start_time) - (CONFIG.startHour * 60);
        const endMins = timeToMinutes(block.end_time) - (CONFIG.startHour * 60);
        const top = (startMins / CONFIG.slotDuration) * CONFIG.slotHeight;
        const height = ((endMins - startMins) / CONFIG.slotDuration) * CONFIG.slotHeight - 2;
        
        const typeIcons = { break: '☕', leave: '🏖️', meeting: '👥', other: '📌' };
        
        const el = document.createElement('div');
        el.className = 'absolute rounded text-xs blocked-slot cursor-pointer border-l-3';
        el.style.cssText = `
            top: ${top}px;
            left: 4px;
            width: ${columnWidth - 12}px;
            height: ${height}px;
            border-left: 3px solid #9ca3af;
        `;
        
        el.innerHTML = `
            <div class="p-1">
                <div class="font-semibold">${typeIcons[block.block_type] || '📌'} ${block.start_time.slice(0,5)}-${block.end_time.slice(0,5)}</div>
                <div class="text-gray-600">${block.reason || 'Bloke'}</div>
            </div>
        `;
        
        el.ondblclick = () => deleteBlock(block.id);
        
        return el;
    }

    // Drag & Drop
    function startDrag(e, appt) {
        e.preventDefault();
        const rect = e.currentTarget.getBoundingClientRect();
        state.dragState = {
            appt,
            offsetY: e.clientY - rect.top,
            startY: e.clientY,
            element: e.currentTarget
        };
        e.currentTarget.classList.add('dragging');
        document.addEventListener('mousemove', onDrag);
        document.addEventListener('mouseup', endDrag);
    }

    function onDrag(e) {
        if (!state.dragState) return;
        const { appt, offsetY, element } = state.dragState;
        
        const grid = element.parentElement;
        const gridRect = grid.getBoundingClientRect();
        const relativeY = e.clientY - gridRect.top - offsetY;
        
        const snapMins = Math.round((relativeY / CONFIG.slotHeight) * CONFIG.slotDuration / CONFIG.slotDuration) * CONFIG.slotDuration;
        const newTop = (snapMins / CONFIG.slotDuration) * CONFIG.slotHeight;
        
        if (newTop >= 0) {
            element.style.top = newTop + 'px';
            state.dragState.newStartMins = CONFIG.startHour * 60 + snapMins;
        }
    }

    function endDrag(e) {
        document.removeEventListener('mousemove', onDrag);
        document.removeEventListener('mouseup', endDrag);
        
        if (!state.dragState) return;
        
        const { appt, element, newStartMins } = state.dragState;
        element.classList.remove('dragging');
        
        if (newStartMins !== undefined) {
            const duration = timeToMinutes(appt.end_time) - timeToMinutes(appt.start_time);
            const newStart = minutesToTime(newStartMins);
            const newEnd = minutesToTime(newStartMins + duration);
            
            updateAppointmentTime(appt.id, newStart, newEnd);
        }
        
        state.dragState = null;
    }

    // Resize
    function startResize(e, appt, direction) {
        e.preventDefault();
        e.stopPropagation();
        state.resizeState = {
            appt,
            direction,
            startY: e.clientY,
            element: e.currentTarget.parentElement
        };
        document.addEventListener('mousemove', onResize);
        document.addEventListener('mouseup', endResize);
    }

    function onResize(e) {
        if (!state.resizeState) return;
        const { appt, direction, startY, element } = state.resizeState;
        
        const deltaY = e.clientY - startY;
        const deltaMins = Math.round((deltaY / CONFIG.slotHeight) * CONFIG.slotDuration / CONFIG.slotDuration) * CONFIG.slotDuration;
        
        if (direction === 'bottom') {
            const newEndMins = timeToMinutes(appt.end_time) + deltaMins;
            const startMins = timeToMinutes(appt.start_time);
            if (newEndMins > startMins + CONFIG.slotDuration) {
                state.resizeState.newEndMins = newEndMins;
                const height = ((newEndMins - startMins) / CONFIG.slotDuration) * CONFIG.slotHeight - 2;
                element.style.height = height + 'px';
            }
        } else {
            const newStartMins = timeToMinutes(appt.start_time) + deltaMins;
            const endMins = timeToMinutes(appt.end_time);
            if (newStartMins < endMins - CONFIG.slotDuration && newStartMins >= CONFIG.startHour * 60) {
                state.resizeState.newStartMins = newStartMins;
                const top = ((newStartMins - CONFIG.startHour * 60) / CONFIG.slotDuration) * CONFIG.slotHeight;
                const height = ((endMins - newStartMins) / CONFIG.slotDuration) * CONFIG.slotHeight - 2;
                element.style.top = top + 'px';
                element.style.height = height + 'px';
            }
        }
    }

    function endResize(e) {
        document.removeEventListener('mousemove', onResize);
        document.removeEventListener('mouseup', endResize);
        
        if (!state.resizeState) return;
        
        const { appt, newStartMins, newEndMins } = state.resizeState;
        const newStart = newStartMins ? minutesToTime(newStartMins) : appt.start_time;
        const newEnd = newEndMins ? minutesToTime(newEndMins) : appt.end_time;
        
        if (newStart !== appt.start_time || newEnd !== appt.end_time) {
            updateAppointmentTime(appt.id, newStart, newEnd);
        }
        
        state.resizeState = null;
    }

    // API: Randevu zamanını güncelle
    async function updateAppointmentTime(id, startTime, endTime) {
        try {
            const response = await fetch(`${CONFIG.apiUrl}/appointments.php`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ id, start_time: startTime, end_time: endTime })
            });
            const data = await response.json();
            if (data.success) {
                loadAppointments();
            } else {
                alert(data.error || 'Hata oluştu');
                loadAppointments();
            }
        } catch (error) {
            console.error('Güncelleme hatası:', error);
            loadAppointments();
        }
    }

    // Modal işlemleri
    function openModal(id) {
        document.getElementById(id).classList.add('active');
    }

    function closeModal(id) {
        document.getElementById(id).classList.remove('active');
    }

    function openAddModal(staffId = null, time = null) {
        document.getElementById('modalTitle').textContent = 'Yeni Randevu';
        document.getElementById('appointmentId').value = '';
        document.getElementById('appointmentForm').reset();
        document.getElementById('deleteBtn').classList.add('hidden');
        
        document.getElementById('appointmentDate').value = state.date;
        if (staffId) document.getElementById('staffId').value = staffId;
        if (time) {
            document.getElementById('startTime').value = time;
            const [h, m] = time.split(':').map(Number);
            document.getElementById('endTime').value = minutesToTime(h * 60 + m + 30);
        }
        
        setPatientMode('existing');
        openModal('appointmentModal');
    }

    function openEditModal(appt) {
        document.getElementById('modalTitle').textContent = 'Randevu Düzenle';
        document.getElementById('appointmentId').value = appt.id;
        document.getElementById('patientId').innerHTML = `<option value="${appt.patient_id}">${appt.patient_name}</option>`;
        document.getElementById('patientId').value = appt.patient_id;
        document.getElementById('staffId').value = appt.staff_id;
        document.getElementById('serviceId').value = appt.service_id;
        document.getElementById('appointmentDate').value = appt.appointment_date;
        document.getElementById('startTime').value = appt.start_time.slice(0, 5);
        document.getElementById('endTime').value = appt.end_time.slice(0, 5);
        document.getElementById('status').value = appt.status;
        document.getElementById('notes').value = appt.notes || '';
        document.getElementById('deleteBtn').classList.remove('hidden');
        
        setPatientMode('existing');
        openModal('appointmentModal');
    }

    function openBlockModal() {
        document.getElementById('blockForm').reset();
        document.getElementById('blockDate').value = state.date;
        document.getElementById('blockStaffId').value = state.selectedStaff;
        openModal('blockModal');
    }

    // Hasta modu
    function setPatientMode(mode) {
        state.patientMode = mode;
        document.getElementById('btnExisting').className = mode === 'existing' ? 'flex-1 py-2 rounded bg-blue-600 text-white text-sm font-medium' : 'flex-1 py-2 rounded bg-gray-100 text-gray-700 text-sm font-medium';
        document.getElementById('btnNew').className = mode === 'new' ? 'flex-1 py-2 rounded bg-blue-600 text-white text-sm font-medium' : 'flex-1 py-2 rounded bg-gray-100 text-gray-700 text-sm font-medium';
        document.getElementById('existingPatientSection').style.display = mode === 'existing' ? 'block' : 'none';
        document.getElementById('newPatientSection').classList.toggle('hidden', mode !== 'new');
    }

    // Hasta arama
    async function searchPatients(query) {
        if (query.length < 2) {
            document.getElementById('patientResults').classList.add('hidden');
            return;
        }
        
        try {
            const response = await fetch(`${CONFIG.apiUrl}/patients.php?search=${encodeURIComponent(query)}`);
            const data = await response.json();
            
            const resultsEl = document.getElementById('patientResults');
            if (data.patients && data.patients.length > 0) {
                resultsEl.innerHTML = data.patients.map(p => `
                    <div class="p-2 hover:bg-gray-50 cursor-pointer border-b" onclick="selectPatient(${p.id}, '${p.first_name} ${p.last_name}')">
                        <div class="font-medium text-sm">${p.first_name} ${p.last_name}</div>
                        <div class="text-xs text-gray-500">${p.phone}</div>
                    </div>
                `).join('');
                resultsEl.classList.remove('hidden');
            } else {
                resultsEl.innerHTML = '<div class="p-3 text-center text-gray-500 text-sm">Hasta bulunamadı</div>';
                resultsEl.classList.remove('hidden');
            }
        } catch (error) {
            console.error('Arama hatası:', error);
        }
    }

    function selectPatient(id, name) {
        document.getElementById('patientId').innerHTML = `<option value="${id}" selected>${name}</option>`;
        document.getElementById('patientId').value = id;
        document.getElementById('patientSearch').value = name;
        document.getElementById('patientResults').classList.add('hidden');
    }

    // Hizmet süresi güncelle
    function updateDuration() {
        const select = document.getElementById('serviceId');
        const option = select.options[select.selectedIndex];
        const duration = option?.dataset?.duration || 30;
        
        const startTime = document.getElementById('startTime').value;
        if (startTime) {
            const [h, m] = startTime.split(':').map(Number);
            document.getElementById('endTime').value = minutesToTime(h * 60 + m + parseInt(duration));
        }
    }

    // Randevu kaydet
    async function saveAppointment(e) {
        e.preventDefault();
        
        const formData = {
            id: document.getElementById('appointmentId').value || null,
            staff_id: document.getElementById('staffId').value,
            service_id: document.getElementById('serviceId').value,
            appointment_date: document.getElementById('appointmentDate').value,
            start_time: document.getElementById('startTime').value,
            end_time: document.getElementById('endTime').value,
            status: document.getElementById('status').value,
            notes: document.getElementById('notes').value
        };
        
        if (state.patientMode === 'new') {
            formData.new_patient = {
                first_name: document.getElementById('newFirstName').value,
                last_name: document.getElementById('newLastName').value,
                phone: document.getElementById('newPhone').value
            };
        } else {
            formData.patient_id = document.getElementById('patientId').value;
        }
        
        try {
            const method = formData.id ? 'PUT' : 'POST';
            const response = await fetch(`${CONFIG.apiUrl}/appointments.php`, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();
            if (data.success) {
                closeModal('appointmentModal');
                loadAppointments();
            } else {
                alert(data.error || 'Hata oluştu');
            }
        } catch (error) {
            console.error('Kayıt hatası:', error);
            alert('Bağlantı hatası');
        }
    }

    // Randevu sil
    async function deleteAppointment() {
        if (!confirm('Bu randevuyu silmek istediğinize emin misiniz?')) return;
        
        const id = document.getElementById('appointmentId').value;
        try {
            const response = await fetch(`${CONFIG.apiUrl}/appointments.php?id=${id}`, { method: 'DELETE' });
            const data = await response.json();
            if (data.success) {
                closeModal('appointmentModal');
                loadAppointments();
            } else {
                alert(data.error || 'Hata oluştu');
            }
        } catch (error) {
            console.error('Silme hatası:', error);
        }
    }

    // Blok kaydet
    async function saveBlock(e) {
        e.preventDefault();
        
        const formData = {
            staff_id: document.getElementById('blockStaffId').value,
            block_type: document.getElementById('blockType').value,
            block_date: document.getElementById('blockDate').value,
            start_time: document.getElementById('blockStartTime').value,
            end_time: document.getElementById('blockEndTime').value,
            reason: document.getElementById('blockReason').value
        };
        
        try {
            const response = await fetch(`${CONFIG.apiUrl}/blocked-times.php`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();
            if (data.success) {
                closeModal('blockModal');
                loadAppointments();
            } else {
                alert(data.error || 'Hata oluştu');
            }
        } catch (error) {
            console.error('Blok kayıt hatası:', error);
        }
    }

    // Blok sil
    async function deleteBlock(id) {
        if (!confirm('Bu bloğu silmek istediğinize emin misiniz?')) return;
        
        try {
            const response = await fetch(`${CONFIG.apiUrl}/blocked-times.php?id=${id}`, { method: 'DELETE' });
            const data = await response.json();
            if (data.success) {
                loadAppointments();
            }
        } catch (error) {
            console.error('Blok silme hatası:', error);
        }
    }

    // Randevu filtrele
    function filterAppointments() {
        const query = document.getElementById('searchInput').value.toLowerCase();
        document.querySelectorAll('.appointment-card').forEach(card => {
            const text = card.textContent.toLowerCase();
            card.style.display = text.includes(query) ? 'block' : 'none';
        });
    }
    </script>
</body>
</html>
