<?php
/**
 * WordPress Migration Script
 * 
 * WordPress veritabanından yeni PHP CMS'e veri aktarımı
 * 
 * KULLANIM:
 * 1. Kurulum tamamlandıktan sonra bu dosyayı çalıştırın
 * 2. https://test.drozguroktaynar.com/migration/migrate.php
 * 3. Adım adım migration işlemlerini seçin
 */

// Load config
require_once __DIR__ . '/../includes/config.php';

// Check if logged in as admin (security)
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    // For initial setup, allow access with a secret key
    $secretKey = $_GET['key'] ?? '';
    if ($secretKey !== 'migrate2024drnar') {
        die('Unauthorized. Add ?key=migrate2024drnar or login as admin first.');
    }
}

class WordPressMigration {
    private $wpDb;
    private $newDb;
    private $wpPrefix = 'wp_';
    private $uploadDir;
    private $logs = [];
    
    public function __construct() {
        // Connect to WordPress database
        try {
            $dsn = "mysql:host=" . WP_DB_HOST . ";dbname=" . WP_DB_NAME . ";charset=utf8mb4";
            $this->wpDb = new PDO($dsn, WP_DB_USER, WP_DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } catch (PDOException $e) {
            die("WordPress DB connection failed: " . $e->getMessage());
        }
        
        // Connect to new database
        $this->newDb = Database::getInstance()->getPdo();
        
        $this->uploadDir = ROOT_PATH . '/uploads';
    }
    
    /**
     * Get WordPress statistics
     */
    public function getStats(): array {
        $stats = [];
        
        // Posts count
        $stmt = $this->wpDb->query("SELECT COUNT(*) FROM {$this->wpPrefix}posts WHERE post_type = 'post' AND post_status = 'publish'");
        $stats['posts'] = $stmt->fetchColumn();
        
        // Pages count
        $stmt = $this->wpDb->query("SELECT COUNT(*) FROM {$this->wpPrefix}posts WHERE post_type = 'page' AND post_status = 'publish'");
        $stats['pages'] = $stmt->fetchColumn();
        
        // Categories count
        $stmt = $this->wpDb->query("SELECT COUNT(*) FROM {$this->wpPrefix}terms t 
            INNER JOIN {$this->wpPrefix}term_taxonomy tt ON t.term_id = tt.term_id 
            WHERE tt.taxonomy = 'category'");
        $stats['categories'] = $stmt->fetchColumn();
        
        // Tags count
        $stmt = $this->wpDb->query("SELECT COUNT(*) FROM {$this->wpPrefix}terms t 
            INNER JOIN {$this->wpPrefix}term_taxonomy tt ON t.term_id = tt.term_id 
            WHERE tt.taxonomy = 'post_tag'");
        $stats['tags'] = $stmt->fetchColumn();
        
        // Media count
        $stmt = $this->wpDb->query("SELECT COUNT(*) FROM {$this->wpPrefix}posts WHERE post_type = 'attachment'");
        $stats['media'] = $stmt->fetchColumn();
        
        return $stats;
    }
    
    /**
     * Migrate categories
     */
    public function migrateCategories(): array {
        $this->log("Starting category migration...");
        
        $stmt = $this->wpDb->query("
            SELECT t.term_id, t.name, t.slug, tt.description, tt.parent, tt.count
            FROM {$this->wpPrefix}terms t
            INNER JOIN {$this->wpPrefix}term_taxonomy tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'category'
            ORDER BY tt.parent, t.name
        ");
        
        $categories = $stmt->fetchAll();
        $migrated = 0;
        $skipped = 0;
        
        foreach ($categories as $cat) {
            // Check if already exists
            $exists = $this->newDb->prepare("SELECT id FROM categories WHERE old_wp_id = ?");
            $exists->execute([$cat['term_id']]);
            
            if ($exists->fetch()) {
                $skipped++;
                continue;
            }
            
            // Find parent_id in new system
            $parentId = null;
            if ($cat['parent'] > 0) {
                $parentStmt = $this->newDb->prepare("SELECT id FROM categories WHERE old_wp_id = ?");
                $parentStmt->execute([$cat['parent']]);
                $parent = $parentStmt->fetch();
                $parentId = $parent ? $parent['id'] : null;
            }
            
            // Insert category
            $insert = $this->newDb->prepare("
                INSERT INTO categories (old_wp_id, slug, name_tr, name_en, description_tr, parent_id, post_count)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $insert->execute([
                $cat['term_id'],
                $cat['slug'],
                $cat['name'],
                $cat['name'], // Same for English initially
                $cat['description'],
                $parentId,
                $cat['count']
            ]);
            
            $migrated++;
        }
        
        $this->log("Categories migrated: {$migrated}, skipped: {$skipped}");
        
        return ['migrated' => $migrated, 'skipped' => $skipped];
    }
    
    /**
     * Migrate tags
     */
    public function migrateTags(): array {
        $this->log("Starting tag migration...");
        
        $stmt = $this->wpDb->query("
            SELECT t.term_id, t.name, t.slug, tt.count
            FROM {$this->wpPrefix}terms t
            INNER JOIN {$this->wpPrefix}term_taxonomy tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'post_tag'
        ");
        
        $tags = $stmt->fetchAll();
        $migrated = 0;
        $skipped = 0;
        
        foreach ($tags as $tag) {
            // Check if already exists
            $exists = $this->newDb->prepare("SELECT id FROM tags WHERE old_wp_id = ?");
            $exists->execute([$tag['term_id']]);
            
            if ($exists->fetch()) {
                $skipped++;
                continue;
            }
            
            $insert = $this->newDb->prepare("
                INSERT INTO tags (old_wp_id, slug, name_tr, name_en, post_count)
                VALUES (?, ?, ?, ?, ?)
            ");
            
            $insert->execute([
                $tag['term_id'],
                $tag['slug'],
                $tag['name'],
                $tag['name'],
                $tag['count']
            ]);
            
            $migrated++;
        }
        
        $this->log("Tags migrated: {$migrated}, skipped: {$skipped}");
        
        return ['migrated' => $migrated, 'skipped' => $skipped];
    }
    
    /**
     * Migrate posts (blog articles)
     */
    public function migratePosts(): array {
        $this->log("Starting post migration...");
        
        $stmt = $this->wpDb->query("
            SELECT p.ID, p.post_title, p.post_content, p.post_excerpt, p.post_name,
                   p.post_status, p.post_date, p.post_modified
            FROM {$this->wpPrefix}posts p
            WHERE p.post_type = 'post' AND p.post_status IN ('publish', 'draft')
            ORDER BY p.post_date DESC
        ");
        
        $posts = $stmt->fetchAll();
        $migrated = 0;
        $skipped = 0;
        
        foreach ($posts as $post) {
            // Check if already exists
            $exists = $this->newDb->prepare("SELECT id FROM posts WHERE old_wp_id = ?");
            $exists->execute([$post['ID']]);
            
            if ($exists->fetch()) {
                $skipped++;
                continue;
            }
            
            // Get category
            $catStmt = $this->wpDb->prepare("
                SELECT tt.term_id
                FROM {$this->wpPrefix}term_relationships tr
                INNER JOIN {$this->wpPrefix}term_taxonomy tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                WHERE tr.object_id = ? AND tt.taxonomy = 'category'
                LIMIT 1
            ");
            $catStmt->execute([$post['ID']]);
            $wpCatId = $catStmt->fetchColumn();
            
            // Get new category ID
            $categoryId = null;
            if ($wpCatId) {
                $newCatStmt = $this->newDb->prepare("SELECT id FROM categories WHERE old_wp_id = ?");
                $newCatStmt->execute([$wpCatId]);
                $categoryId = $newCatStmt->fetchColumn();
            }
            
            // Get SEO meta from Yoast
            $metaTitle = $this->getPostMeta($post['ID'], '_yoast_wpseo_title');
            $metaDesc = $this->getPostMeta($post['ID'], '_yoast_wpseo_metadesc');
            $focusKeyword = $this->getPostMeta($post['ID'], '_yoast_wpseo_focuskw');
            
            // Get featured image
            $thumbnailId = $this->getPostMeta($post['ID'], '_thumbnail_id');
            $featuredImage = null;
            if ($thumbnailId) {
                $imgStmt = $this->wpDb->prepare("SELECT guid FROM {$this->wpPrefix}posts WHERE ID = ?");
                $imgStmt->execute([$thumbnailId]);
                $featuredImage = $imgStmt->fetchColumn();
                // Convert to relative path
                if ($featuredImage) {
                    $featuredImage = str_replace('https://www.drozguroktaynar.com/wp-content/uploads/', '', $featuredImage);
                    $featuredImage = str_replace('https://drozguroktaynar.com/wp-content/uploads/', '', $featuredImage);
                }
            }
            
            // Clean content
            $content = $this->cleanWordPressContent($post['post_content']);
            
            // Calculate reading time
            $readingTime = max(1, ceil(str_word_count(strip_tags($content)) / 200));
            
            // Insert post
            $insert = $this->newDb->prepare("
                INSERT INTO posts (
                    old_wp_id, slug, title_tr, title_en, content_tr, content_en,
                    excerpt_tr, excerpt_en, meta_title_tr, meta_description_tr,
                    meta_keywords_tr, featured_image, category_id, reading_time,
                    status, created_at, updated_at, published_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $status = $post['post_status'] === 'publish' ? 'published' : 'draft';
            
            $insert->execute([
                $post['ID'],
                $post['post_name'],
                $post['post_title'],
                $post['post_title'], // Same for English initially
                $content,
                $content,
                $post['post_excerpt'] ?: truncate($content, 200, ''),
                $post['post_excerpt'] ?: truncate($content, 200, ''),
                $metaTitle ?: $post['post_title'],
                $metaDesc ?: truncate($content, 160, ''),
                $focusKeyword,
                $featuredImage,
                $categoryId,
                $readingTime,
                $status,
                $post['post_date'],
                $post['post_modified'],
                $status === 'published' ? $post['post_date'] : null
            ]);
            
            $newPostId = $this->newDb->lastInsertId();
            
            // Migrate post tags
            $this->migratePostTags($post['ID'], $newPostId);
            
            $migrated++;
        }
        
        $this->log("Posts migrated: {$migrated}, skipped: {$skipped}");
        
        return ['migrated' => $migrated, 'skipped' => $skipped];
    }
    
    /**
     * Migrate post tags relationship
     */
    private function migratePostTags(int $wpPostId, int $newPostId): void {
        $stmt = $this->wpDb->prepare("
            SELECT tt.term_id
            FROM {$this->wpPrefix}term_relationships tr
            INNER JOIN {$this->wpPrefix}term_taxonomy tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
            WHERE tr.object_id = ? AND tt.taxonomy = 'post_tag'
        ");
        $stmt->execute([$wpPostId]);
        
        while ($row = $stmt->fetch()) {
            // Get new tag ID
            $tagStmt = $this->newDb->prepare("SELECT id FROM tags WHERE old_wp_id = ?");
            $tagStmt->execute([$row['term_id']]);
            $newTagId = $tagStmt->fetchColumn();
            
            if ($newTagId) {
                try {
                    $insert = $this->newDb->prepare("INSERT INTO post_tags (post_id, tag_id) VALUES (?, ?)");
                    $insert->execute([$newPostId, $newTagId]);
                } catch (PDOException $e) {
                    // Duplicate, ignore
                }
            }
        }
    }
    
    /**
     * Migrate pages
     */
    public function migratePages(): array {
        $this->log("Starting page migration...");
        
        $stmt = $this->wpDb->query("
            SELECT p.ID, p.post_title, p.post_content, p.post_excerpt, p.post_name,
                   p.post_status, p.post_date, p.post_modified, p.post_parent, p.menu_order
            FROM {$this->wpPrefix}posts p
            WHERE p.post_type = 'page' AND p.post_status IN ('publish', 'draft')
            ORDER BY p.post_parent, p.menu_order
        ");
        
        $pages = $stmt->fetchAll();
        $migrated = 0;
        $skipped = 0;
        
        foreach ($pages as $page) {
            // Check if already exists
            $exists = $this->newDb->prepare("SELECT id FROM pages WHERE old_wp_id = ?");
            $exists->execute([$page['ID']]);
            
            if ($exists->fetch()) {
                $skipped++;
                continue;
            }
            
            // Get parent ID
            $parentId = null;
            if ($page['post_parent'] > 0) {
                $parentStmt = $this->newDb->prepare("SELECT id FROM pages WHERE old_wp_id = ?");
                $parentStmt->execute([$page['post_parent']]);
                $parentId = $parentStmt->fetchColumn();
            }
            
            // Get SEO meta
            $metaTitle = $this->getPostMeta($page['ID'], '_yoast_wpseo_title');
            $metaDesc = $this->getPostMeta($page['ID'], '_yoast_wpseo_metadesc');
            
            // Get featured image
            $thumbnailId = $this->getPostMeta($page['ID'], '_thumbnail_id');
            $featuredImage = null;
            if ($thumbnailId) {
                $imgStmt = $this->wpDb->prepare("SELECT guid FROM {$this->wpPrefix}posts WHERE ID = ?");
                $imgStmt->execute([$thumbnailId]);
                $featuredImage = $imgStmt->fetchColumn();
                if ($featuredImage) {
                    $featuredImage = str_replace(['https://www.drozguroktaynar.com/wp-content/uploads/', 'https://drozguroktaynar.com/wp-content/uploads/'], '', $featuredImage);
                }
            }
            
            // Determine template based on slug
            $template = 'default';
            if ($page['post_name'] === 'iletisim' || $page['post_name'] === 'contact') {
                $template = 'contact';
            } elseif ($page['post_name'] === 'hakkimda' || $page['post_name'] === 'about') {
                $template = 'about';
            } elseif (strpos($page['post_name'], 'blog') !== false) {
                $template = 'blog';
            }
            
            // Clean content
            $content = $this->cleanWordPressContent($page['post_content']);
            
            $insert = $this->newDb->prepare("
                INSERT INTO pages (
                    old_wp_id, slug, title_tr, title_en, content_tr, content_en,
                    excerpt_tr, meta_title_tr, meta_description_tr, featured_image,
                    template, parent_id, sort_order, status, created_at, updated_at, published_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $status = $page['post_status'] === 'publish' ? 'published' : 'draft';
            
            $insert->execute([
                $page['ID'],
                $page['post_name'],
                $page['post_title'],
                $page['post_title'],
                $content,
                $content,
                $page['post_excerpt'] ?: truncate($content, 200, ''),
                $metaTitle ?: $page['post_title'],
                $metaDesc ?: truncate($content, 160, ''),
                $featuredImage,
                $template,
                $parentId,
                $page['menu_order'],
                $status,
                $page['post_date'],
                $page['post_modified'],
                $status === 'published' ? $page['post_date'] : null
            ]);
            
            $migrated++;
        }
        
        $this->log("Pages migrated: {$migrated}, skipped: {$skipped}");
        
        return ['migrated' => $migrated, 'skipped' => $skipped];
    }
    
    /**
     * Generate 301 redirects for old URLs
     */
    public function generateRedirects(): array {
        $this->log("Generating redirects...");
        $count = 0;
        
        // Posts
        $stmt = $this->newDb->query("SELECT old_wp_id, slug FROM posts WHERE old_wp_id IS NOT NULL");
        while ($row = $stmt->fetch()) {
            // Get old permalink
            $oldUrl = '/' . date('Y/m', strtotime($this->getWpPostDate($row['old_wp_id']))) . '/' . $row['slug'] . '/';
            $newUrl = '/blog/' . $row['slug'];
            
            $this->insertRedirect($oldUrl, $newUrl);
            $count++;
        }
        
        // Pages
        $stmt = $this->newDb->query("SELECT old_wp_id, slug FROM pages WHERE old_wp_id IS NOT NULL");
        while ($row = $stmt->fetch()) {
            $oldUrl = '/' . $row['slug'] . '/';
            $newUrl = '/' . $row['slug'];
            
            if ($oldUrl !== $newUrl) {
                $this->insertRedirect($oldUrl, $newUrl);
                $count++;
            }
        }
        
        $this->log("Redirects generated: {$count}");
        
        return ['generated' => $count];
    }
    
    /**
     * Helper: Get post meta
     */
    private function getPostMeta(int $postId, string $key) {
        $stmt = $this->wpDb->prepare("SELECT meta_value FROM {$this->wpPrefix}postmeta WHERE post_id = ? AND meta_key = ?");
        $stmt->execute([$postId, $key]);
        return $stmt->fetchColumn();
    }
    
    /**
     * Helper: Get WordPress post date
     */
    private function getWpPostDate(int $postId): string {
        $stmt = $this->wpDb->prepare("SELECT post_date FROM {$this->wpPrefix}posts WHERE ID = ?");
        $stmt->execute([$postId]);
        return $stmt->fetchColumn() ?: date('Y-m-d');
    }
    
    /**
     * Helper: Insert redirect
     */
    private function insertRedirect(string $oldUrl, string $newUrl): void {
        try {
            $stmt = $this->newDb->prepare("INSERT INTO redirects (old_url, new_url) VALUES (?, ?) ON DUPLICATE KEY UPDATE new_url = VALUES(new_url)");
            $stmt->execute([$oldUrl, $newUrl]);
        } catch (PDOException $e) {
            // Ignore duplicates
        }
    }
    
    /**
     * Clean WordPress content
     */
    private function cleanWordPressContent(string $content): string {
        // Remove WordPress shortcodes
        $content = preg_replace('/\[.*?\]/', '', $content);
        
        // Remove Elementor/Gutenberg JSON data
        $content = preg_replace('/<!-- wp:.*?-->.*?<!-- \/wp:.*?-->/s', '', $content);
        
        // Remove extra whitespace
        $content = preg_replace('/\s+/', ' ', $content);
        $content = preg_replace('/<p>\s*<\/p>/', '', $content);
        
        // Fix image URLs
        $content = str_replace(
            ['https://www.drozguroktaynar.com/wp-content/uploads/', 'https://drozguroktaynar.com/wp-content/uploads/'],
            '/uploads/',
            $content
        );
        
        return trim($content);
    }
    
    /**
     * Log message
     */
    private function log(string $message): void {
        $this->logs[] = date('H:i:s') . ' - ' . $message;
    }
    
    public function getLogs(): array {
        return $this->logs;
    }
}

// Handle AJAX requests
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    $migration = new WordPressMigration();
    
    switch ($_GET['action']) {
        case 'stats':
            echo json_encode($migration->getStats());
            break;
            
        case 'categories':
            echo json_encode($migration->migrateCategories());
            break;
            
        case 'tags':
            echo json_encode($migration->migrateTags());
            break;
            
        case 'posts':
            echo json_encode($migration->migratePosts());
            break;
            
        case 'pages':
            echo json_encode($migration->migratePages());
            break;
            
        case 'redirects':
            echo json_encode($migration->generateRedirects());
            break;
            
        default:
            echo json_encode(['error' => 'Unknown action']);
    }
    exit;
}

// Show migration interface
$migration = new WordPressMigration();
$stats = $migration->getStats();
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WordPress Migration - Dr. Özgür Oktay Nar</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Inter', sans-serif;
            background: #f1f5f9;
            padding: 40px 20px;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
        }
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);
            padding: 24px;
            margin-bottom: 20px;
        }
        h1 {
            color: #1e40af;
            margin-bottom: 8px;
        }
        h2 {
            color: #334155;
            font-size: 18px;
            margin-bottom: 16px;
        }
        .subtitle {
            color: #64748b;
            margin-bottom: 24px;
        }
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }
        .stat {
            background: #f8fafc;
            padding: 16px;
            border-radius: 8px;
            text-align: center;
        }
        .stat-value {
            font-size: 28px;
            font-weight: 700;
            color: #0ea5e9;
        }
        .stat-label {
            font-size: 13px;
            color: #64748b;
            margin-top: 4px;
        }
        .migration-step {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 16px;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            margin-bottom: 12px;
        }
        .step-info h3 {
            font-size: 15px;
            color: #334155;
            margin-bottom: 4px;
        }
        .step-info p {
            font-size: 13px;
            color: #94a3b8;
        }
        .btn {
            padding: 10px 20px;
            background: linear-gradient(135deg, #0ea5e9, #1e40af);
            color: white;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        .btn:hover { transform: translateY(-1px); }
        .btn:disabled {
            background: #94a3b8;
            cursor: not-allowed;
            transform: none;
        }
        .btn.success {
            background: #10b981;
        }
        .status {
            font-size: 13px;
            padding: 4px 10px;
            border-radius: 20px;
            margin-left: 12px;
        }
        .status.pending { background: #fef3c7; color: #92400e; }
        .status.running { background: #dbeafe; color: #1e40af; }
        .status.done { background: #d1fae5; color: #065f46; }
        .status.error { background: #fee2e2; color: #991b1b; }
        .log {
            background: #1e293b;
            color: #e2e8f0;
            padding: 16px;
            border-radius: 8px;
            font-family: monospace;
            font-size: 13px;
            max-height: 300px;
            overflow-y: auto;
            margin-top: 20px;
        }
        .log-entry {
            padding: 4px 0;
            border-bottom: 1px solid #334155;
        }
        .all-done {
            background: #d1fae5;
            border: 2px solid #10b981;
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            color: #065f46;
        }
        .all-done h3 {
            font-size: 20px;
            margin-bottom: 8px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <h1>🚀 WordPress Migration</h1>
            <p class="subtitle">WordPress veritabanından yeni PHP CMS'e veri aktarımı</p>
            
            <h2>📊 WordPress İstatistikleri</h2>
            <div class="stats">
                <div class="stat">
                    <div class="stat-value"><?= $stats['posts'] ?></div>
                    <div class="stat-label">Blog Yazısı</div>
                </div>
                <div class="stat">
                    <div class="stat-value"><?= $stats['pages'] ?></div>
                    <div class="stat-label">Sayfa</div>
                </div>
                <div class="stat">
                    <div class="stat-value"><?= $stats['categories'] ?></div>
                    <div class="stat-label">Kategori</div>
                </div>
                <div class="stat">
                    <div class="stat-value"><?= $stats['tags'] ?></div>
                    <div class="stat-label">Etiket</div>
                </div>
                <div class="stat">
                    <div class="stat-value"><?= $stats['media'] ?></div>
                    <div class="stat-label">Medya</div>
                </div>
            </div>
        </div>
        
        <div class="card">
            <h2>📦 Migration Adımları</h2>
            
            <div class="migration-step" id="step-categories">
                <div class="step-info">
                    <h3>1. Kategoriler</h3>
                    <p>Blog kategorilerini aktar</p>
                </div>
                <div>
                    <span class="status pending" id="status-categories">Bekliyor</span>
                    <button class="btn" onclick="runMigration('categories')">Başlat</button>
                </div>
            </div>
            
            <div class="migration-step" id="step-tags">
                <div class="step-info">
                    <h3>2. Etiketler</h3>
                    <p>Blog etiketlerini aktar</p>
                </div>
                <div>
                    <span class="status pending" id="status-tags">Bekliyor</span>
                    <button class="btn" onclick="runMigration('tags')">Başlat</button>
                </div>
            </div>
            
            <div class="migration-step" id="step-posts">
                <div class="step-info">
                    <h3>3. Blog Yazıları</h3>
                    <p>Tüm blog yazılarını ve SEO metalarını aktar</p>
                </div>
                <div>
                    <span class="status pending" id="status-posts">Bekliyor</span>
                    <button class="btn" onclick="runMigration('posts')">Başlat</button>
                </div>
            </div>
            
            <div class="migration-step" id="step-pages">
                <div class="step-info">
                    <h3>4. Sayfalar</h3>
                    <p>Statik sayfaları aktar (Hakkımda, İletişim vb.)</p>
                </div>
                <div>
                    <span class="status pending" id="status-pages">Bekliyor</span>
                    <button class="btn" onclick="runMigration('pages')">Başlat</button>
                </div>
            </div>
            
            <div class="migration-step" id="step-redirects">
                <div class="step-info">
                    <h3>5. 301 Yönlendirmeler</h3>
                    <p>Eski URL'ler için SEO-friendly yönlendirmeler oluştur</p>
                </div>
                <div>
                    <span class="status pending" id="status-redirects">Bekliyor</span>
                    <button class="btn" onclick="runMigration('redirects')">Başlat</button>
                </div>
            </div>
            
            <button class="btn" style="width: 100%; margin-top: 20px; padding: 14px;" onclick="runAllMigrations()">
                🚀 Tümünü Çalıştır
            </button>
            
            <div class="log" id="log" style="display: none;"></div>
        </div>
        
        <div class="card all-done" id="all-done" style="display: none;">
            <h3>✅ Migration Tamamlandı!</h3>
            <p>Tüm veriler başarıyla aktarıldı. Şimdi admin panelden kontrol edebilirsiniz.</p>
            <br>
            <a href="/admin" class="btn success">Admin Panel'e Git</a>
        </div>
    </div>
    
    <script>
        const steps = ['categories', 'tags', 'posts', 'pages', 'redirects'];
        let completedSteps = [];
        
        function log(message) {
            const logEl = document.getElementById('log');
            logEl.style.display = 'block';
            logEl.innerHTML += `<div class="log-entry">${new Date().toLocaleTimeString()} - ${message}</div>`;
            logEl.scrollTop = logEl.scrollHeight;
        }
        
        async function runMigration(action) {
            const statusEl = document.getElementById('status-' + action);
            const btn = event.target;
            
            statusEl.className = 'status running';
            statusEl.textContent = 'Çalışıyor...';
            btn.disabled = true;
            
            log(`Starting ${action} migration...`);
            
            try {
                const response = await fetch(`?action=${action}&key=migrate2024drnar`);
                const result = await response.json();
                
                statusEl.className = 'status done';
                statusEl.textContent = `✓ ${result.migrated || result.generated || 0} aktarıldı`;
                btn.textContent = '✓';
                btn.classList.add('success');
                
                completedSteps.push(action);
                log(`${action} completed: ${JSON.stringify(result)}`);
                
                checkAllDone();
            } catch (error) {
                statusEl.className = 'status error';
                statusEl.textContent = 'Hata!';
                btn.disabled = false;
                log(`Error in ${action}: ${error.message}`);
            }
        }
        
        async function runAllMigrations() {
            for (const step of steps) {
                if (!completedSteps.includes(step)) {
                    const btn = document.querySelector(`#step-${step} .btn`);
                    btn.click();
                    await new Promise(r => setTimeout(r, 2000)); // Wait between steps
                }
            }
        }
        
        function checkAllDone() {
            if (completedSteps.length === steps.length) {
                document.getElementById('all-done').style.display = 'block';
            }
        }
    </script>
</body>
</html>
