<?php
/**
 * Helper Functions
 */

// =====================================================
// STRING & TEXT FUNCTIONS
// =====================================================

/**
 * Generate URL-friendly slug
 */
function slugify(string $text, string $lang = 'tr'): string {
    // Turkish character map
    $turkish = [
        'ş' => 's', 'Ş' => 's', 'ı' => 'i', 'İ' => 'i',
        'ğ' => 'g', 'Ğ' => 'g', 'ü' => 'u', 'Ü' => 'u',
        'ö' => 'o', 'Ö' => 'o', 'ç' => 'c', 'Ç' => 'c'
    ];
    
    $text = strtr($text, $turkish);
    $text = mb_strtolower($text, 'UTF-8');
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    $text = trim($text, '-');
    
    return $text;
}

/**
 * Truncate text with ellipsis
 */
function truncate(string $text, int $length = 150, string $append = '...'): string {
    $text = strip_tags($text);
    if (mb_strlen($text) <= $length) {
        return $text;
    }
    
    $text = mb_substr($text, 0, $length);
    $lastSpace = mb_strrpos($text, ' ');
    
    if ($lastSpace !== false) {
        $text = mb_substr($text, 0, $lastSpace);
    }
    
    return $text . $append;
}

/**
 * Calculate reading time in minutes
 */
function readingTime(string $content, int $wordsPerMinute = 200): int {
    $wordCount = str_word_count(strip_tags($content));
    return max(1, ceil($wordCount / $wordsPerMinute));
}

/**
 * Clean HTML content
 */
function cleanHtml(string $html): string {
    // Remove WordPress shortcodes
    $html = preg_replace('/\[.*?\]/', '', $html);
    
    // Remove empty paragraphs
    $html = preg_replace('/<p>\s*<\/p>/', '', $html);
    
    // Clean up whitespace
    $html = preg_replace('/\s+/', ' ', $html);
    
    return trim($html);
}

/**
 * Extract plain text from HTML
 */
function htmlToText(string $html): string {
    $text = strip_tags($html);
    $text = html_entity_decode($text, ENT_QUOTES, 'UTF-8');
    $text = preg_replace('/\s+/', ' ', $text);
    return trim($text);
}

// =====================================================
// URL & PATH FUNCTIONS
// =====================================================

/**
 * Generate full URL
 */
function url(string $path = ''): string {
    $path = ltrim($path, '/');
    return SITE_URL . '/' . $path;
}

/**
 * Generate asset URL
 */
function asset(string $path): string {
    $path = ltrim($path, '/');
    return ASSETS_URL . '/' . $path;
}

/**
 * Generate upload URL
 */
function uploadUrl(string $path): string {
    $path = ltrim($path, '/');
    return UPLOADS_URL . '/' . $path;
}

/**
 * Get current language
 */
function currentLang(): string {
    return $_SESSION['lang'] ?? DEFAULT_LANG;
}

/**
 * Generate language-aware URL
 */
function langUrl(string $path, string $lang = null): string {
    $lang = $lang ?? currentLang();
    
    if ($lang === DEFAULT_LANG) {
        return url($path);
    }
    
    return url($lang . '/' . ltrim($path, '/'));
}

/**
 * Switch language URL
 */
function switchLangUrl(string $targetLang): string {
    $currentPath = $_SERVER['REQUEST_URI'];
    $currentLang = currentLang();
    
    // Remove current language prefix if exists
    if ($currentLang !== DEFAULT_LANG) {
        $currentPath = preg_replace('/^\/' . $currentLang . '/', '', $currentPath);
    }
    
    // Add new language prefix if not default
    if ($targetLang !== DEFAULT_LANG) {
        return url($targetLang . $currentPath);
    }
    
    return url(ltrim($currentPath, '/'));
}

// =====================================================
// LOCALIZATION FUNCTIONS
// =====================================================

/**
 * Get localized field value
 */
function localized(array $data, string $field, string $lang = null): string {
    $lang = $lang ?? currentLang();
    $key = $field . '_' . $lang;
    
    // Try requested language
    if (!empty($data[$key])) {
        return $data[$key];
    }
    
    // Fallback to default language
    $defaultKey = $field . '_' . DEFAULT_LANG;
    if (!empty($data[$defaultKey])) {
        return $data[$defaultKey];
    }
    
    // Fallback to field without suffix
    return $data[$field] ?? '';
}

/**
 * Translation function (basic)
 */
function __($key, $lang = null): string {
    static $translations = [];
    $lang = $lang ?? currentLang();
    
    if (!isset($translations[$lang])) {
        $file = ROOT_PATH . '/languages/' . $lang . '.php';
        if (file_exists($file)) {
            $translations[$lang] = include $file;
        } else {
            $translations[$lang] = [];
        }
    }
    
    return $translations[$lang][$key] ?? $key;
}

// =====================================================
// DATE & TIME FUNCTIONS
// =====================================================

/**
 * Format date for display
 */
function formatDate($date, string $format = null): string {
    if (empty($date)) {
        return '';
    }
    
    $format = $format ?? getSetting('date_format', 'd.m.Y');
    
    if (is_string($date)) {
        $date = new DateTime($date);
    }
    
    return $date->format($format);
}

/**
 * Format datetime for display
 */
function formatDateTime($date, string $format = null): string {
    if (empty($date)) {
        return '';
    }
    
    $dateFormat = getSetting('date_format', 'd.m.Y');
    $timeFormat = getSetting('time_format', 'H:i');
    $format = $format ?? $dateFormat . ' ' . $timeFormat;
    
    if (is_string($date)) {
        $date = new DateTime($date);
    }
    
    return $date->format($format);
}

/**
 * Human readable time ago
 */
function timeAgo($datetime): string {
    if (is_string($datetime)) {
        $datetime = new DateTime($datetime);
    }
    
    $now = new DateTime();
    $diff = $now->diff($datetime);
    
    if ($diff->y > 0) return $diff->y . ' yıl önce';
    if ($diff->m > 0) return $diff->m . ' ay önce';
    if ($diff->d > 0) return $diff->d . ' gün önce';
    if ($diff->h > 0) return $diff->h . ' saat önce';
    if ($diff->i > 0) return $diff->i . ' dakika önce';
    
    return 'Az önce';
}

// =====================================================
// SECURITY FUNCTIONS
// =====================================================

/**
 * Generate CSRF token
 */
function csrfToken(): string {
    if (empty($_SESSION['csrf_token']) || empty($_SESSION['csrf_token_time']) || 
        time() - $_SESSION['csrf_token_time'] > CSRF_TOKEN_LIFETIME) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        $_SESSION['csrf_token_time'] = time();
    }
    
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 */
function verifyCsrf(string $token): bool {
    if (empty($_SESSION['csrf_token'])) {
        return false;
    }
    
    return hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Generate CSRF input field
 */
function csrfField(): string {
    return '<input type="hidden" name="csrf_token" value="' . csrfToken() . '">';
}

/**
 * Sanitize input
 */
function sanitize($input) {
    if (is_array($input)) {
        return array_map('sanitize', $input);
    }
    
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate email
 */
function isValidEmail(string $email): bool {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate URL
 */
function isValidUrl(string $url): bool {
    return filter_var($url, FILTER_VALIDATE_URL) !== false;
}

/**
 * Hash password
 */
function hashPassword(string $password): string {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Verify password
 */
function verifyPassword(string $password, string $hash): bool {
    return password_verify($password, $hash);
}

/**
 * Generate random string
 */
function randomString(int $length = 32): string {
    return bin2hex(random_bytes($length / 2));
}

// =====================================================
// FILE FUNCTIONS
// =====================================================

/**
 * Get file extension
 */
function getExtension(string $filename): string {
    return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
}

/**
 * Check if file type is allowed
 */
function isAllowedFileType(string $mimeType): bool {
    return in_array($mimeType, ALLOWED_FILE_TYPES);
}

/**
 * Format file size for display
 */
function formatFileSize(int $bytes): string {
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = 0;
    
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    
    return round($bytes, 2) . ' ' . $units[$i];
}

/**
 * Create directory if not exists
 */
function ensureDirectory(string $path): bool {
    if (!is_dir($path)) {
        return mkdir($path, 0755, true);
    }
    return true;
}

// =====================================================
// IMAGE FUNCTIONS
// =====================================================

/**
 * Get image dimensions
 */
function getImageDimensions(string $path): ?array {
    if (!file_exists($path)) {
        return null;
    }
    
    $info = getimagesize($path);
    
    if ($info === false) {
        return null;
    }
    
    return [
        'width' => $info[0],
        'height' => $info[1],
        'mime' => $info['mime']
    ];
}

/**
 * Generate responsive image srcset
 */
function imageSrcset(string $path, array $sizes = [320, 640, 1024, 1920]): string {
    $srcset = [];
    $ext = getExtension($path);
    $base = str_replace('.' . $ext, '', $path);
    
    foreach ($sizes as $size) {
        $thumbPath = "{$base}-{$size}w.{$ext}";
        if (file_exists(UPLOADS_PATH . '/' . $thumbPath)) {
            $srcset[] = uploadUrl($thumbPath) . " {$size}w";
        }
    }
    
    return implode(', ', $srcset);
}

// =====================================================
// SEO FUNCTIONS
// =====================================================

/**
 * Generate meta title
 */
function metaTitle(string $pageTitle = '', string $separator = null): string {
    $separator = $separator ?? getSetting('meta_title_separator', ' | ');
    $siteName = getSetting('site_name_' . currentLang(), SITE_NAME);
    
    if (empty($pageTitle)) {
        return $siteName;
    }
    
    return $pageTitle . $separator . $siteName;
}

/**
 * Generate canonical URL
 */
function canonicalUrl(string $path = ''): string {
    if (empty($path)) {
        $path = $_SERVER['REQUEST_URI'];
    }
    
    // Remove query string
    $path = strtok($path, '?');
    
    // Remove trailing slash except for homepage
    if ($path !== '/') {
        $path = rtrim($path, '/');
    }
    
    return SITE_URL . $path;
}

/**
 * Generate JSON-LD schema
 */
function schemaOrg(array $data): string {
    return '<script type="application/ld+json">' . json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
}

// =====================================================
// CACHE FUNCTIONS
// =====================================================

/**
 * Get cached data
 */
function cacheGet(string $key) {
    if (!CACHE_ENABLED) {
        return null;
    }
    
    $file = CACHE_PATH . '/' . md5($key) . '.cache';
    
    if (!file_exists($file)) {
        return null;
    }
    
    $data = unserialize(file_get_contents($file));
    
    if ($data['expires'] < time()) {
        unlink($file);
        return null;
    }
    
    return $data['value'];
}

/**
 * Set cached data
 */
function cacheSet(string $key, $value, int $ttl = null): bool {
    if (!CACHE_ENABLED) {
        return false;
    }
    
    $ttl = $ttl ?? CACHE_TTL;
    $file = CACHE_PATH . '/' . md5($key) . '.cache';
    
    $data = [
        'value' => $value,
        'expires' => time() + $ttl
    ];
    
    return file_put_contents($file, serialize($data)) !== false;
}

/**
 * Delete cached data
 */
function cacheDelete(string $key): bool {
    $file = CACHE_PATH . '/' . md5($key) . '.cache';
    
    if (file_exists($file)) {
        return unlink($file);
    }
    
    return true;
}

/**
 * Clear all cache
 */
function cacheClear(): int {
    $count = 0;
    $files = glob(CACHE_PATH . '/*.cache');
    
    foreach ($files as $file) {
        if (unlink($file)) {
            $count++;
        }
    }
    
    return $count;
}

// =====================================================
// RESPONSE FUNCTIONS
// =====================================================

/**
 * Send JSON response
 */
function jsonResponse(array $data, int $statusCode = 200): void {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Redirect to URL
 */
function redirect(string $url, int $statusCode = 302): void {
    header('Location: ' . $url, true, $statusCode);
    exit;
}

/**
 * Redirect back
 */
function redirectBack(): void {
    $referer = $_SERVER['HTTP_REFERER'] ?? url('/');
    redirect($referer);
}

/**
 * Set flash message
 */
function setFlash(string $type, string $message): void {
    $_SESSION['flash'][$type] = $message;
}

/**
 * Get and clear flash message
 */
function getFlash(string $type): ?string {
    if (isset($_SESSION['flash'][$type])) {
        $message = $_SESSION['flash'][$type];
        unset($_SESSION['flash'][$type]);
        return $message;
    }
    return null;
}

/**
 * Check if has flash message
 */
function hasFlash(string $type): bool {
    return isset($_SESSION['flash'][$type]);
}

// =====================================================
// DEBUG FUNCTIONS
// =====================================================

/**
 * Debug dump
 */
function dd(...$vars): void {
    echo '<pre style="background:#1e1e1e;color:#fff;padding:20px;margin:20px;border-radius:8px;overflow:auto;">';
    foreach ($vars as $var) {
        var_dump($var);
        echo "\n";
    }
    echo '</pre>';
    exit;
}

/**
 * Log message
 */
function logMessage(string $message, string $level = 'info', array $context = []): void {
    $levels = ['debug', 'info', 'warning', 'error'];
    $configLevel = array_search(LOG_LEVEL, $levels);
    $messageLevel = array_search($level, $levels);
    
    if ($messageLevel < $configLevel) {
        return;
    }
    
    $logFile = LOGS_PATH . '/' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    
    $contextStr = empty($context) ? '' : ' ' . json_encode($context);
    $logEntry = "[{$timestamp}] [{$level}] {$message}{$contextStr}\n";
    
    ensureDirectory(LOGS_PATH);
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}
