<?php
/**
 * Simple Router Class
 * Clean URL routing for the CMS
 */

class Router {
    private $routes = [];
    private $notFoundCallback;
    private $basePath = '';
    
    public function __construct(string $basePath = '') {
        $this->basePath = $basePath;
    }
    
    /**
     * Add a GET route
     */
    public function get(string $pattern, callable $callback): self {
        $this->addRoute('GET', $pattern, $callback);
        return $this;
    }
    
    /**
     * Add a POST route
     */
    public function post(string $pattern, callable $callback): self {
        $this->addRoute('POST', $pattern, $callback);
        return $this;
    }
    
    /**
     * Add route for any method
     */
    public function any(string $pattern, callable $callback): self {
        $this->addRoute('GET', $pattern, $callback);
        $this->addRoute('POST', $pattern, $callback);
        return $this;
    }
    
    /**
     * Add route
     */
    private function addRoute(string $method, string $pattern, callable $callback): void {
        // Convert route pattern to regex
        $pattern = $this->basePath . $pattern;
        
        // Replace {param} with named capture groups
        $pattern = preg_replace('/\{([a-zA-Z_]+)\}/', '(?P<$1>[^/]+)', $pattern);
        
        // Replace {param?} with optional named capture groups
        $pattern = preg_replace('/\{([a-zA-Z_]+)\?\}/', '(?P<$1>[^/]*)?', $pattern);
        
        $this->routes[] = [
            'method' => $method,
            'pattern' => '#^' . $pattern . '$#',
            'callback' => $callback
        ];
    }
    
    /**
     * Set 404 handler
     */
    public function notFound(callable $callback): self {
        $this->notFoundCallback = $callback;
        return $this;
    }
    
    /**
     * Run the router
     */
    public function run(): void {
        $method = $_SERVER['REQUEST_METHOD'];
        $uri = $this->getUri();
        
        // Check redirects first
        if ($this->handleRedirects($uri)) {
            return;
        }
        
        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }
            
            if (preg_match($route['pattern'], $uri, $matches)) {
                // Filter only named parameters
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
                
                call_user_func_array($route['callback'], $params);
                return;
            }
        }
        
        // No route matched
        if ($this->notFoundCallback) {
            http_response_code(404);
            call_user_func($this->notFoundCallback);
        } else {
            http_response_code(404);
            echo '404 Not Found';
        }
    }
    
    /**
     * Get clean URI
     */
    private function getUri(): string {
        $uri = $_SERVER['REQUEST_URI'];
        
        // Remove query string
        if (false !== $pos = strpos($uri, '?')) {
            $uri = substr($uri, 0, $pos);
        }
        
        // Remove trailing slash (except for root)
        $uri = rawurldecode($uri);
        if ($uri !== '/') {
            $uri = rtrim($uri, '/');
        }
        
        return $uri;
    }
    
    /**
     * Handle database redirects
     */
    private function handleRedirects(string $uri): bool {
        global $db;
        
        try {
            $stmt = $db->query(
                "SELECT new_url, redirect_type FROM redirects WHERE old_url = ? AND status = 'active' LIMIT 1",
                [$uri]
            );
            
            $redirect = $stmt->fetch();
            
            if ($redirect) {
                // Update hit count
                $db->query(
                    "UPDATE redirects SET hit_count = hit_count + 1, last_hit = NOW() WHERE old_url = ?",
                    [$uri]
                );
                
                $code = (int) $redirect['redirect_type'];
                header('Location: ' . $redirect['new_url'], true, $code);
                return true;
            }
        } catch (Exception $e) {
            // Table might not exist yet
        }
        
        return false;
    }
}

/**
 * View renderer helper
 */
class View {
    private static $data = [];
    private static $themePath;
    
    public static function init(string $themePath): void {
        self::$themePath = $themePath;
    }
    
    /**
     * Set data for views
     */
    public static function set(string $key, $value): void {
        self::$data[$key] = $value;
    }
    
    /**
     * Get data
     */
    public static function get(string $key, $default = null) {
        return self::$data[$key] ?? $default;
    }
    
    /**
     * Render a template
     */
    public static function render(string $template, array $data = []): void {
        $data = array_merge(self::$data, $data);
        extract($data);
        
        $templateFile = self::$themePath . '/templates/' . $template . '.php';
        
        if (!file_exists($templateFile)) {
            throw new Exception("Template not found: {$template}");
        }
        
        // Start output buffering
        ob_start();
        
        include $templateFile;
        
        echo ob_get_clean();
    }
    
    /**
     * Include a partial
     */
    public static function partial(string $name, array $data = []): void {
        $data = array_merge(self::$data, $data);
        extract($data);
        
        $file = self::$themePath . '/templates/' . $name . '.php';
        
        if (file_exists($file)) {
            include $file;
        }
    }
}
