<?php
/**
 * n8n Webhook API Endpoint
 * 
 * Bu endpoint n8n workflow'larından gelen içerik verilerini alır
 * ve veritabanına kaydeder.
 * 
 * WEBHOOK URL: https://test.drozguroktaynar.com/api/webhook.php
 */

// CORS headers (n8n için gerekli)
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Webhook-Secret');
header('Content-Type: application/json');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

require_once __DIR__ . '/../includes/config.php';

// Verify webhook secret (optional but recommended)
$secret = $_SERVER['HTTP_X_WEBHOOK_SECRET'] ?? '';
$configSecret = N8N_WEBHOOK_SECRET;

if (!empty($configSecret) && $secret !== $configSecret) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Get JSON payload
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON']);
    exit;
}

// Log incoming request
logMessage('n8n webhook received', 'info', $data);

try {
    $action = $data['action'] ?? 'create_content';
    
    switch ($action) {
        
        case 'create_content':
            // AI tarafından üretilen içeriği kuyruğa ekle
            $result = createContent($data);
            break;
            
        case 'publish_content':
            // Onaylanan içeriği yayınla
            $result = publishContent($data);
            break;
            
        case 'get_keywords':
            // Trend keywordleri al
            $result = getKeywords();
            break;
            
        case 'update_reviews':
            // Google yorumlarını güncelle
            $result = updateReviews($data);
            break;
            
        case 'sync_analytics':
            // Analytics verilerini senkronize et
            $result = syncAnalytics($data);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Unknown action: ' . $action]);
            exit;
    }
    
    echo json_encode([
        'success' => true,
        'data' => $result
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
    
    logMessage('n8n webhook error: ' . $e->getMessage(), 'error');
}

// =====================================================
// ACTION HANDLERS
// =====================================================

/**
 * Create content from AI generation
 */
function createContent(array $data): array {
    global $db;
    
    $required = ['title', 'content', 'keyword'];
    foreach ($required as $field) {
        if (empty($data[$field])) {
            throw new Exception("Missing required field: {$field}");
        }
    }
    
    // Prepare meta data
    $meta = [
        'meta_title' => $data['meta_title'] ?? null,
        'meta_description' => $data['meta_description'] ?? null,
        'tags' => $data['tags'] ?? [],
        'category' => $data['category'] ?? null,
        'featured_image_prompt' => $data['image_prompt'] ?? null,
    ];
    
    // Insert into queue
    $id = $db->insert('n8n_content_queue', [
        'workflow_id' => $data['workflow_id'] ?? null,
        'execution_id' => $data['execution_id'] ?? null,
        'content_type' => $data['type'] ?? 'post',
        'source_keyword' => $data['keyword'],
        'generated_title' => $data['title'],
        'generated_content' => $data['content'],
        'generated_meta' => json_encode($meta),
        'ai_model' => $data['model'] ?? 'claude-3-sonnet',
        'status' => 'pending'
    ]);
    
    return [
        'id' => $id,
        'message' => 'Content added to queue',
        'status' => 'pending',
        'review_url' => SITE_URL . '/admin/ai-content.php?action=review&id=' . $id
    ];
}

/**
 * Publish approved content
 */
function publishContent(array $data): array {
    global $db;
    
    if (empty($data['queue_id'])) {
        throw new Exception("Missing queue_id");
    }
    
    // Get queue item
    $queue = $db->getRow("SELECT * FROM n8n_content_queue WHERE id = ?", [$data['queue_id']]);
    
    if (!$queue) {
        throw new Exception("Queue item not found");
    }
    
    if ($queue['status'] !== 'approved') {
        throw new Exception("Content is not approved");
    }
    
    $meta = json_decode($queue['generated_meta'], true);
    $lang = currentLang();
    
    // Create slug
    $slug = slugify($queue['generated_title']);
    
    // Check for duplicate slug
    $counter = 1;
    $originalSlug = $slug;
    while ($db->exists('posts', 'slug = ?', [$slug])) {
        $slug = $originalSlug . '-' . $counter;
        $counter++;
    }
    
    // Get category ID
    $categoryId = null;
    if (!empty($meta['category'])) {
        $category = $db->getRow("SELECT id FROM categories WHERE slug = ? OR name_tr = ?", 
            [$meta['category'], $meta['category']]);
        $categoryId = $category['id'] ?? null;
    }
    
    // Insert post
    $postId = $db->insert('posts', [
        'slug' => $slug,
        'title_tr' => $queue['generated_title'],
        'title_en' => $queue['generated_title'],
        'content_tr' => $queue['generated_content'],
        'content_en' => $queue['generated_content'],
        'excerpt_tr' => truncate(strip_tags($queue['generated_content']), 200),
        'meta_title_tr' => $meta['meta_title'] ?? $queue['generated_title'],
        'meta_description_tr' => $meta['meta_description'] ?? truncate(strip_tags($queue['generated_content']), 160),
        'category_id' => $categoryId,
        'reading_time' => readingTime($queue['generated_content']),
        'status' => 'published',
        'ai_generated' => 1,
        'human_reviewed' => 1,
        'ai_model' => $queue['ai_model'],
        'source_keyword' => $queue['source_keyword'],
        'published_at' => date('Y-m-d H:i:s')
    ]);
    
    // Update queue
    $db->update('n8n_content_queue', [
        'status' => 'published',
        'published_as_id' => $postId
    ], 'id = ?', ['id' => $queue['id']]);
    
    // Handle tags
    if (!empty($meta['tags'])) {
        foreach ($meta['tags'] as $tagName) {
            $tagSlug = slugify($tagName);
            
            // Find or create tag
            $tag = $db->getRow("SELECT id FROM tags WHERE slug = ?", [$tagSlug]);
            
            if (!$tag) {
                $tagId = $db->insert('tags', [
                    'slug' => $tagSlug,
                    'name_tr' => $tagName,
                    'name_en' => $tagName
                ]);
            } else {
                $tagId = $tag['id'];
            }
            
            // Link tag to post
            try {
                $db->insert('post_tags', [
                    'post_id' => $postId,
                    'tag_id' => $tagId
                ]);
            } catch (Exception $e) {
                // Duplicate, ignore
            }
        }
    }
    
    return [
        'post_id' => $postId,
        'slug' => $slug,
        'url' => SITE_URL . '/blog/' . $slug,
        'message' => 'Content published successfully'
    ];
}

/**
 * Get trending keywords
 */
function getKeywords(): array {
    global $db;
    
    // Get categories for context
    $categories = $db->getAll("SELECT slug, name_tr FROM categories WHERE status = 'active'");
    
    // Get recent keywords that were used
    $usedKeywords = $db->getAll("
        SELECT source_keyword, COUNT(*) as count 
        FROM posts 
        WHERE source_keyword IS NOT NULL 
        GROUP BY source_keyword 
        ORDER BY count DESC 
        LIMIT 50
    ");
    
    // Suggest related medical keywords (can be enhanced with Google Trends API)
    $suggestedKeywords = [
        'prp tedavisi', 'kök hücre tedavisi', 'proloterapi',
        'diz ağrısı tedavisi', 'bel fıtığı tedavisi', 'omuz ağrısı',
        'spor yaralanmaları', 'menisküs yırtığı', 'çapraz bağ yaralanması',
        'osteoartrit tedavisi', 'boyun fıtığı', 'kalça protezi',
        'diz protezi', 'artroskopi', 'ortopedik rehabilitasyon'
    ];
    
    return [
        'categories' => $categories,
        'used_keywords' => $usedKeywords,
        'suggested_keywords' => $suggestedKeywords
    ];
}

/**
 * Update Google reviews cache
 */
function updateReviews(array $data): array {
    global $db;
    
    if (empty($data['reviews']) || !is_array($data['reviews'])) {
        throw new Exception("No reviews provided");
    }
    
    $inserted = 0;
    $updated = 0;
    
    foreach ($data['reviews'] as $review) {
        $exists = $db->getRow("SELECT id FROM reviews WHERE external_id = ?", [$review['id']]);
        
        $reviewData = [
            'external_id' => $review['id'],
            'author_name' => $review['author_name'] ?? 'Anonim',
            'author_photo' => $review['author_photo'] ?? null,
            'rating' => $review['rating'] ?? 5,
            'text' => $review['text'] ?? '',
            'language' => $review['language'] ?? 'tr',
            'review_time' => $review['time'] ?? null,
            'relative_time' => $review['relative_time'] ?? null,
            'source' => $review['source'] ?? 'google',
            'cached_at' => date('Y-m-d H:i:s')
        ];
        
        if ($exists) {
            $db->update('reviews', $reviewData, 'id = ?', ['id' => $exists['id']]);
            $updated++;
        } else {
            $db->insert('reviews', $reviewData);
            $inserted++;
        }
    }
    
    return [
        'inserted' => $inserted,
        'updated' => $updated,
        'total' => count($data['reviews'])
    ];
}

/**
 * Sync analytics data
 */
function syncAnalytics(array $data): array {
    global $db;
    
    if (empty($data['metrics']) || !is_array($data['metrics'])) {
        throw new Exception("No metrics provided");
    }
    
    foreach ($data['metrics'] as $metric) {
        // Delete old cache for this metric type
        $db->delete('analytics_cache', 'metric_type = ? AND date_range = ?', 
            [$metric['type'], $metric['range']]);
        
        // Insert new cache
        $db->insert('analytics_cache', [
            'metric_type' => $metric['type'],
            'metric_key' => $metric['key'] ?? null,
            'metric_value' => json_encode($metric['value']),
            'date_range' => $metric['range'],
            'start_date' => $metric['start_date'] ?? null,
            'end_date' => $metric['end_date'] ?? null,
            'expires_at' => date('Y-m-d H:i:s', strtotime('+1 hour'))
        ]);
    }
    
    return [
        'synced' => count($data['metrics']),
        'message' => 'Analytics data cached'
    ];
}
