/**
 * Admin Panel JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    // Sidebar toggle
    const sidebarToggle = document.querySelector('.sidebar-toggle');
    if (sidebarToggle) {
        sidebarToggle.addEventListener('click', function() {
            document.body.classList.toggle('sidebar-collapsed');
            localStorage.setItem('sidebar-collapsed', document.body.classList.contains('sidebar-collapsed'));
        });
        
        // Restore state
        if (localStorage.getItem('sidebar-collapsed') === 'true') {
            document.body.classList.add('sidebar-collapsed');
        }
    }
    
    // Auto-hide alerts
    document.querySelectorAll('.alert').forEach(function(alert) {
        setTimeout(function() {
            alert.style.opacity = '0';
            alert.style.transform = 'translateY(-10px)';
            setTimeout(function() {
                alert.remove();
            }, 300);
        }, 5000);
    });
    
    // Confirm delete actions
    document.querySelectorAll('[data-confirm]').forEach(function(el) {
        el.addEventListener('click', function(e) {
            if (!confirm(this.dataset.confirm || 'Bu işlemi gerçekleştirmek istediğinize emin misiniz?')) {
                e.preventDefault();
            }
        });
    });
    
    // Auto-slug generation
    const titleInputs = document.querySelectorAll('[data-slug-source]');
    titleInputs.forEach(function(input) {
        input.addEventListener('input', function() {
            const slugInput = document.querySelector('[name="slug"]');
            if (slugInput && !slugInput.dataset.edited) {
                slugInput.value = slugify(this.value);
            }
        });
    });
    
    // Mark slug as edited if user changes it manually
    const slugInputs = document.querySelectorAll('[name="slug"]');
    slugInputs.forEach(function(input) {
        input.addEventListener('input', function() {
            this.dataset.edited = 'true';
        });
    });
    
    // Image preview
    document.querySelectorAll('input[type="file"][accept*="image"]').forEach(function(input) {
        input.addEventListener('change', function(e) {
            const preview = document.getElementById(this.dataset.preview);
            if (preview && this.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                };
                reader.readAsDataURL(this.files[0]);
            }
        });
    });
    
    // TinyMCE init (if loaded)
    if (typeof tinymce !== 'undefined') {
        tinymce.init({
            selector: 'textarea.wysiwyg',
            height: 400,
            plugins: 'anchor autolink charmap codesample emoticons image link lists media searchreplace table visualblocks wordcount',
            toolbar: 'undo redo | blocks fontfamily fontsize | bold italic underline strikethrough | link image media table | align lineheight | numlist bullist indent outdent | emoticons charmap | removeformat',
            content_style: 'body { font-family: Inter, sans-serif; font-size: 14px; }',
            images_upload_url: '/api/upload.php',
            automatic_uploads: true
        });
    }
});

/**
 * Turkish-friendly slug generator
 */
function slugify(text) {
    const turkish = {
        'ş': 's', 'Ş': 's', 'ı': 'i', 'İ': 'i',
        'ğ': 'g', 'Ğ': 'g', 'ü': 'u', 'Ü': 'u',
        'ö': 'o', 'Ö': 'o', 'ç': 'c', 'Ç': 'c'
    };
    
    for (let key in turkish) {
        text = text.replace(new RegExp(key, 'g'), turkish[key]);
    }
    
    return text
        .toLowerCase()
        .replace(/[^a-z0-9\s-]/g, '')
        .replace(/[\s-]+/g, '-')
        .replace(/^-+|-+$/g, '');
}

/**
 * Ajax form submit
 */
function submitForm(form, callback) {
    const formData = new FormData(form);
    
    fetch(form.action, {
        method: form.method,
        body: formData
    })
    .then(response => response.json())
    .then(data => callback(null, data))
    .catch(error => callback(error));
}

/**
 * Toast notification
 */
function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        bottom: 20px;
        right: 20px;
        padding: 14px 24px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        z-index: 9999;
        animation: slideIn 0.3s ease;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#f59e0b'};
    `;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.opacity = '0';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}
