<?php
/**
 * Dr. Özgür Oktay Nar - PHP CMS Installer
 * 
 * Bu dosyayı sunucunuza yükleyin ve tarayıcıdan açın.
 * Tüm kurulum otomatik olarak yapılacaktır.
 * 
 * KULLANIM:
 * 1. Bu dosyayı ve proje ZIP'ini public_html/test.drozguroktaynar.com/ klasörüne yükleyin
 * 2. ZIP'i extract edin
 * 3. https://test.drozguroktaynar.com/installer.php adresini açın
 * 4. Kurulum tamamlandıktan sonra installer.php dosyasını silin!
 */

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Increase limits for large operations
ini_set('max_execution_time', 300);
ini_set('memory_limit', '512M');

session_start();

// =====================================================
// CONFIGURATION
// =====================================================

$config = [
    // Database settings (Hozzt)
    'db_host' => 'localhost',
    'db_name' => 'onarcom_drnar',          // cPanel'de oluşturulacak
    'db_user' => 'onarcom_drnar',          // cPanel'de oluşturulacak
    'db_pass' => '',                        // Formdan alınacak
    
    // Site settings
    'site_url' => 'https://test.drozguroktaynar.com',
    'admin_email' => 'ozguroktaynar@gmail.com',
    'admin_password' => '',                 // Formdan alınacak
    
    // WordPress migration (Güzel Hosting)
    'wp_db_host' => 'localhost',
    'wp_db_name' => 'drozguroktaynar_12dd',
    'wp_db_user' => 'drozguroktaynar_21ddd',
    'wp_db_pass' => 'zz4613629zz',
    
    // Paths
    'install_path' => __DIR__,
];

// =====================================================
// INSTALLER CLASS
// =====================================================

class Installer {
    private $config;
    private $db;
    private $errors = [];
    private $messages = [];
    
    public function __construct(array $config) {
        $this->config = $config;
    }
    
    /**
     * Check system requirements
     */
    public function checkRequirements(): array {
        $requirements = [];
        
        // PHP Version
        $requirements['php_version'] = [
            'name' => 'PHP Version',
            'required' => '8.0+',
            'current' => PHP_VERSION,
            'status' => version_compare(PHP_VERSION, '8.0.0', '>=')
        ];
        
        // PDO Extension
        $requirements['pdo'] = [
            'name' => 'PDO Extension',
            'required' => 'Enabled',
            'current' => extension_loaded('pdo') ? 'Enabled' : 'Disabled',
            'status' => extension_loaded('pdo')
        ];
        
        // PDO MySQL
        $requirements['pdo_mysql'] = [
            'name' => 'PDO MySQL',
            'required' => 'Enabled',
            'current' => extension_loaded('pdo_mysql') ? 'Enabled' : 'Disabled',
            'status' => extension_loaded('pdo_mysql')
        ];
        
        // JSON Extension
        $requirements['json'] = [
            'name' => 'JSON Extension',
            'required' => 'Enabled',
            'current' => extension_loaded('json') ? 'Enabled' : 'Disabled',
            'status' => extension_loaded('json')
        ];
        
        // Mbstring Extension
        $requirements['mbstring'] = [
            'name' => 'Mbstring Extension',
            'required' => 'Enabled',
            'current' => extension_loaded('mbstring') ? 'Enabled' : 'Disabled',
            'status' => extension_loaded('mbstring')
        ];
        
        // GD or Imagick
        $requirements['image'] = [
            'name' => 'GD or Imagick',
            'required' => 'One of them',
            'current' => extension_loaded('gd') ? 'GD' : (extension_loaded('imagick') ? 'Imagick' : 'None'),
            'status' => extension_loaded('gd') || extension_loaded('imagick')
        ];
        
        // Writable directories
        $writableDirs = ['uploads', 'cache', 'backups', 'logs'];
        foreach ($writableDirs as $dir) {
            $path = $this->config['install_path'] . '/' . $dir;
            $writable = is_dir($path) ? is_writable($path) : is_writable($this->config['install_path']);
            $requirements['writable_' . $dir] = [
                'name' => "Writable: /{$dir}",
                'required' => 'Writable',
                'current' => $writable ? 'Writable' : 'Not Writable',
                'status' => $writable
            ];
        }
        
        return $requirements;
    }
    
    /**
     * Test database connection
     */
    public function testConnection(array $dbConfig): bool {
        try {
            $dsn = "mysql:host={$dbConfig['host']};charset=utf8mb4";
            $pdo = new PDO($dsn, $dbConfig['user'], $dbConfig['pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
            ]);
            
            // Check if database exists
            $stmt = $pdo->query("SHOW DATABASES LIKE '{$dbConfig['name']}'");
            if ($stmt->rowCount() === 0) {
                $this->errors[] = "Veritabanı '{$dbConfig['name']}' bulunamadı. Lütfen cPanel'den oluşturun.";
                return false;
            }
            
            return true;
        } catch (PDOException $e) {
            $this->errors[] = "Veritabanı bağlantı hatası: " . $e->getMessage();
            return false;
        }
    }
    
    /**
     * Create database tables
     */
    public function createTables(): bool {
        try {
            $dsn = "mysql:host={$this->config['db_host']};dbname={$this->config['db_name']};charset=utf8mb4";
            $this->db = new PDO($dsn, $this->config['db_user'], $this->config['db_pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
            ]);
            
            // Read SQL file
            $sqlFile = $this->config['install_path'] . '/database.sql';
            if (!file_exists($sqlFile)) {
                $this->errors[] = "database.sql dosyası bulunamadı!";
                return false;
            }
            
            $sql = file_get_contents($sqlFile);
            
            // Execute SQL
            $this->db->exec($sql);
            
            $this->messages[] = "✅ Veritabanı tabloları oluşturuldu";
            return true;
            
        } catch (PDOException $e) {
            $this->errors[] = "Tablo oluşturma hatası: " . $e->getMessage();
            return false;
        }
    }
    
    /**
     * Update admin password
     */
    public function updateAdminPassword(string $password): bool {
        try {
            $hash = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $this->db->prepare("UPDATE admins SET password_hash = ? WHERE username = 'admin'");
            $stmt->execute([$hash]);
            
            $this->messages[] = "✅ Admin şifresi güncellendi";
            return true;
        } catch (PDOException $e) {
            $this->errors[] = "Admin şifre güncelleme hatası: " . $e->getMessage();
            return false;
        }
    }
    
    /**
     * Update config file
     */
    public function updateConfigFile(): bool {
        $configFile = $this->config['install_path'] . '/includes/config.php';
        
        if (!file_exists($configFile)) {
            $this->errors[] = "config.php dosyası bulunamadı!";
            return false;
        }
        
        $content = file_get_contents($configFile);
        
        // Update database settings
        $content = preg_replace(
            "/define\('DB_PASS', '.*?'\);/",
            "define('DB_PASS', '{$this->config['db_pass']}');",
            $content
        );
        
        // Update site URL
        $content = preg_replace(
            "/define\('SITE_URL', '.*?'\);/",
            "define('SITE_URL', '{$this->config['site_url']}');",
            $content
        );
        
        // Generate and update secret key
        $secretKey = bin2hex(random_bytes(16));
        $content = preg_replace(
            "/define\('SECRET_KEY', '.*?'\);/",
            "define('SECRET_KEY', '{$secretKey}');",
            $content
        );
        
        if (file_put_contents($configFile, $content)) {
            $this->messages[] = "✅ Config dosyası güncellendi";
            return true;
        }
        
        $this->errors[] = "Config dosyası yazılamadı!";
        return false;
    }
    
    /**
     * Create required directories
     */
    public function createDirectories(): bool {
        $dirs = [
            'uploads',
            'uploads/images',
            'uploads/documents',
            'uploads/videos',
            'cache',
            'backups',
            'logs',
            'languages'
        ];
        
        foreach ($dirs as $dir) {
            $path = $this->config['install_path'] . '/' . $dir;
            if (!is_dir($path)) {
                if (!mkdir($path, 0755, true)) {
                    $this->errors[] = "Klasör oluşturulamadı: {$dir}";
                    return false;
                }
            }
            
            // Create .htaccess to protect sensitive directories
            if (in_array($dir, ['backups', 'logs', 'cache'])) {
                $htaccess = $path . '/.htaccess';
                if (!file_exists($htaccess)) {
                    file_put_contents($htaccess, "Deny from all\n");
                }
            }
        }
        
        $this->messages[] = "✅ Klasörler oluşturuldu";
        return true;
    }
    
    /**
     * Create .htaccess file
     */
    public function createHtaccess(): bool {
        $htaccess = $this->config['install_path'] . '/public/.htaccess';
        
        $content = <<<HTACCESS
# Dr. Özgür Oktay Nar - PHP CMS
# Apache Rewrite Rules

<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteBase /
    
    # Force HTTPS
    RewriteCond %{HTTPS} off
    RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
    
    # Remove trailing slash
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]
    
    # Handle front controller
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^(.*)$ index.php?route=$1 [QSA,L]
</IfModule>

# Security Headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options "nosniff"
    Header always set X-Frame-Options "SAMEORIGIN"
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# Protect sensitive files
<FilesMatch "^(\.htaccess|\.htpasswd|config\.php|database\.sql)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# Compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript application/json
</IfModule>

# Caching
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/webp "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
    ExpiresByType font/woff2 "access plus 1 year"
</IfModule>
HTACCESS;
        
        if (file_put_contents($htaccess, $content)) {
            $this->messages[] = "✅ .htaccess oluşturuldu";
            return true;
        }
        
        $this->errors[] = ".htaccess oluşturulamadı!";
        return false;
    }
    
    /**
     * Run full installation
     */
    public function install(): bool {
        // Create directories
        if (!$this->createDirectories()) return false;
        
        // Create tables
        if (!$this->createTables()) return false;
        
        // Update admin password
        if (!$this->updateAdminPassword($this->config['admin_password'])) return false;
        
        // Update config
        if (!$this->updateConfigFile()) return false;
        
        // Create htaccess
        if (!$this->createHtaccess()) return false;
        
        $this->messages[] = "🎉 Kurulum başarıyla tamamlandı!";
        return true;
    }
    
    public function getErrors(): array {
        return $this->errors;
    }
    
    public function getMessages(): array {
        return $this->messages;
    }
}

// =====================================================
// HANDLE FORM SUBMISSION
// =====================================================

$installer = new Installer($config);
$step = $_GET['step'] ?? 'requirements';
$requirements = $installer->checkRequirements();
$allPassed = !in_array(false, array_column($requirements, 'status'));

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'test_connection':
                $dbConfig = [
                    'host' => $_POST['db_host'],
                    'name' => $_POST['db_name'],
                    'user' => $_POST['db_user'],
                    'pass' => $_POST['db_pass']
                ];
                
                if ($installer->testConnection($dbConfig)) {
                    $_SESSION['db_config'] = $dbConfig;
                    header('Location: ?step=install');
                    exit;
                }
                $step = 'database';
                break;
                
            case 'install':
                $config['db_host'] = $_SESSION['db_config']['host'];
                $config['db_name'] = $_SESSION['db_config']['name'];
                $config['db_user'] = $_SESSION['db_config']['user'];
                $config['db_pass'] = $_SESSION['db_config']['pass'];
                $config['admin_password'] = $_POST['admin_password'];
                $config['site_url'] = $_POST['site_url'];
                
                $installer = new Installer($config);
                
                if ($installer->install()) {
                    $step = 'complete';
                } else {
                    $step = 'install';
                }
                break;
        }
    }
}

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kurulum - Dr. Özgür Oktay Nar CMS</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #0ea5e9 0%, #1e40af 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .installer {
            background: white;
            border-radius: 16px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            max-width: 600px;
            width: 100%;
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #0ea5e9 0%, #1e40af 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 24px;
            font-weight: 700;
            margin-bottom: 8px;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 14px;
        }
        
        .steps {
            display: flex;
            justify-content: center;
            gap: 40px;
            padding: 20px;
            background: #f8fafc;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .step {
            display: flex;
            align-items: center;
            gap: 8px;
            color: #94a3b8;
            font-size: 14px;
        }
        
        .step.active {
            color: #0ea5e9;
            font-weight: 600;
        }
        
        .step.completed {
            color: #10b981;
        }
        
        .step-number {
            width: 28px;
            height: 28px;
            border-radius: 50%;
            background: #e2e8f0;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            font-size: 13px;
        }
        
        .step.active .step-number {
            background: #0ea5e9;
            color: white;
        }
        
        .step.completed .step-number {
            background: #10b981;
            color: white;
        }
        
        .content {
            padding: 30px;
        }
        
        .requirement {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 8px;
            background: #f8fafc;
        }
        
        .requirement.pass {
            background: #ecfdf5;
        }
        
        .requirement.fail {
            background: #fef2f2;
        }
        
        .requirement-name {
            font-weight: 500;
            color: #334155;
        }
        
        .requirement-value {
            font-size: 13px;
            color: #64748b;
        }
        
        .status {
            font-size: 20px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 6px;
            font-weight: 500;
            color: #334155;
            font-size: 14px;
        }
        
        .form-group input {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 15px;
            transition: border-color 0.2s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #0ea5e9;
        }
        
        .form-group small {
            display: block;
            margin-top: 4px;
            color: #94a3b8;
            font-size: 12px;
        }
        
        .btn {
            display: inline-block;
            padding: 14px 28px;
            background: linear-gradient(135deg, #0ea5e9 0%, #1e40af 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
            width: 100%;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(14, 165, 233, 0.3);
        }
        
        .btn:disabled {
            background: #94a3b8;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        .alert {
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert-error {
            background: #fef2f2;
            color: #991b1b;
            border: 1px solid #fecaca;
        }
        
        .alert-success {
            background: #ecfdf5;
            color: #065f46;
            border: 1px solid #a7f3d0;
        }
        
        .messages {
            margin-top: 20px;
        }
        
        .message {
            padding: 10px 16px;
            border-radius: 6px;
            margin-bottom: 8px;
            font-size: 14px;
        }
        
        .message.success {
            background: #ecfdf5;
            color: #065f46;
        }
        
        .message.error {
            background: #fef2f2;
            color: #991b1b;
        }
        
        .complete-icon {
            width: 80px;
            height: 80px;
            background: #10b981;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
            font-size: 40px;
            color: white;
        }
        
        .complete-text {
            text-align: center;
        }
        
        .complete-text h2 {
            color: #10b981;
            margin-bottom: 10px;
        }
        
        .complete-text p {
            color: #64748b;
            margin-bottom: 20px;
        }
        
        .links {
            display: flex;
            gap: 12px;
            margin-top: 20px;
        }
        
        .links a {
            flex: 1;
            text-align: center;
            padding: 14px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            transition: background 0.2s;
        }
        
        .links a.primary {
            background: linear-gradient(135deg, #0ea5e9 0%, #1e40af 100%);
            color: white;
        }
        
        .links a.secondary {
            background: #f1f5f9;
            color: #334155;
        }
        
        .warning-box {
            background: #fffbeb;
            border: 1px solid #fcd34d;
            border-radius: 8px;
            padding: 16px;
            margin-top: 20px;
            font-size: 13px;
            color: #92400e;
        }
        
        .warning-box strong {
            display: block;
            margin-bottom: 4px;
        }
    </style>
</head>
<body>
    <div class="installer">
        <div class="header">
            <h1>🏥 Dr. Özgür Oktay Nar</h1>
            <p>PHP CMS Kurulum Sihirbazı</p>
        </div>
        
        <div class="steps">
            <div class="step <?php echo $step === 'requirements' ? 'active' : ($step !== 'requirements' ? 'completed' : ''); ?>">
                <span class="step-number">1</span>
                Gereksinimler
            </div>
            <div class="step <?php echo $step === 'database' ? 'active' : (in_array($step, ['install', 'complete']) ? 'completed' : ''); ?>">
                <span class="step-number">2</span>
                Veritabanı
            </div>
            <div class="step <?php echo $step === 'install' ? 'active' : ($step === 'complete' ? 'completed' : ''); ?>">
                <span class="step-number">3</span>
                Kurulum
            </div>
        </div>
        
        <div class="content">
            <?php if ($step === 'requirements'): ?>
                <h3 style="margin-bottom: 20px; color: #334155;">Sistem Gereksinimleri</h3>
                
                <?php foreach ($requirements as $req): ?>
                    <div class="requirement <?php echo $req['status'] ? 'pass' : 'fail'; ?>">
                        <div>
                            <div class="requirement-name"><?php echo $req['name']; ?></div>
                            <div class="requirement-value">Gerekli: <?php echo $req['required']; ?> | Mevcut: <?php echo $req['current']; ?></div>
                        </div>
                        <span class="status"><?php echo $req['status'] ? '✅' : '❌'; ?></span>
                    </div>
                <?php endforeach; ?>
                
                <?php if ($allPassed): ?>
                    <a href="?step=database" class="btn" style="display: block; text-align: center; text-decoration: none; margin-top: 20px;">
                        Devam Et →
                    </a>
                <?php else: ?>
                    <div class="alert alert-error" style="margin-top: 20px;">
                        Lütfen yukarıdaki gereksinimleri karşılayın ve sayfayı yenileyin.
                    </div>
                <?php endif; ?>
                
            <?php elseif ($step === 'database'): ?>
                <h3 style="margin-bottom: 20px; color: #334155;">Veritabanı Ayarları</h3>
                
                <?php if ($installer->getErrors()): ?>
                    <div class="alert alert-error">
                        <?php echo implode('<br>', $installer->getErrors()); ?>
                    </div>
                <?php endif; ?>
                
                <div class="alert alert-success" style="margin-bottom: 20px;">
                    <strong>📋 cPanel'de yapmanız gerekenler:</strong><br>
                    1. MySQL Databases bölümüne gidin<br>
                    2. "onarcom_drnar" adında veritabanı oluşturun<br>
                    3. Aynı isimde kullanıcı oluşturun ve şifre belirleyin<br>
                    4. Kullanıcıyı veritabanına "ALL PRIVILEGES" ile ekleyin
                </div>
                
                <form method="POST">
                    <input type="hidden" name="action" value="test_connection">
                    
                    <div class="form-group">
                        <label>Veritabanı Sunucusu</label>
                        <input type="text" name="db_host" value="localhost" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Veritabanı Adı</label>
                        <input type="text" name="db_name" value="onarcom_drnar" required>
                        <small>cPanel'de "onarcom_" prefix'i otomatik eklenir</small>
                    </div>
                    
                    <div class="form-group">
                        <label>Veritabanı Kullanıcısı</label>
                        <input type="text" name="db_user" value="onarcom_drnar" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Veritabanı Şifresi</label>
                        <input type="password" name="db_pass" required>
                        <small>cPanel'de belirlediğiniz şifre</small>
                    </div>
                    
                    <button type="submit" class="btn">Bağlantıyı Test Et →</button>
                </form>
                
            <?php elseif ($step === 'install'): ?>
                <h3 style="margin-bottom: 20px; color: #334155;">Kurulum Ayarları</h3>
                
                <?php if ($installer->getErrors()): ?>
                    <div class="alert alert-error">
                        <?php echo implode('<br>', $installer->getErrors()); ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST">
                    <input type="hidden" name="action" value="install">
                    
                    <div class="form-group">
                        <label>Site URL</label>
                        <input type="url" name="site_url" value="https://test.drozguroktaynar.com" required>
                        <small>Sonunda / olmasın</small>
                    </div>
                    
                    <div class="form-group">
                        <label>Admin Şifresi</label>
                        <input type="password" name="admin_password" required minlength="8">
                        <small>En az 8 karakter. Kullanıcı adı: admin</small>
                    </div>
                    
                    <button type="submit" class="btn">🚀 Kurulumu Başlat</button>
                </form>
                
                <?php if ($installer->getMessages()): ?>
                    <div class="messages">
                        <?php foreach ($installer->getMessages() as $msg): ?>
                            <div class="message success"><?php echo $msg; ?></div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
                
            <?php elseif ($step === 'complete'): ?>
                <div class="complete-icon">✓</div>
                <div class="complete-text">
                    <h2>Kurulum Tamamlandı!</h2>
                    <p>Siteniz kullanıma hazır. Aşağıdaki linklerden erişebilirsiniz.</p>
                </div>
                
                <div class="messages">
                    <?php foreach ($installer->getMessages() as $msg): ?>
                        <div class="message success"><?php echo $msg; ?></div>
                    <?php endforeach; ?>
                </div>
                
                <div class="links">
                    <a href="<?php echo $config['site_url']; ?>/admin" class="primary">Admin Panel</a>
                    <a href="<?php echo $config['site_url']; ?>" class="secondary">Siteyi Görüntüle</a>
                </div>
                
                <div class="warning-box">
                    <strong>⚠️ Güvenlik Uyarısı</strong>
                    Kurulum tamamlandıktan sonra bu dosyayı (installer.php) sunucudan SİLMEYİ unutmayın!
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
