<?php
/**
 * Admin Panel - Posts Management
 */

require_once __DIR__ . '/../includes/config.php';

// Check authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: /admin/login.php');
    exit;
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Filters
$status = $_GET['status'] ?? '';
$category = $_GET['category'] ?? '';
$search = $_GET['search'] ?? '';

// Build query
$where = [];
$params = [];

if ($status) {
    $where[] = "p.status = ?";
    $params[] = $status;
}

if ($category) {
    $where[] = "p.category_id = ?";
    $params[] = $category;
}

if ($search) {
    $where[] = "(p.title_tr LIKE ? OR p.title_en LIKE ?)";
    $params[] = "%{$search}%";
    $params[] = "%{$search}%";
}

$whereClause = $where ? "WHERE " . implode(" AND ", $where) : "";

// Get total count
$countSql = "SELECT COUNT(*) FROM posts p {$whereClause}";
$total = $db->getValue($countSql, $params);
$totalPages = ceil($total / $perPage);

// Get posts
$sql = "SELECT p.*, c.name_tr as category_name 
        FROM posts p 
        LEFT JOIN categories c ON p.category_id = c.id 
        {$whereClause}
        ORDER BY p.created_at DESC 
        LIMIT {$perPage} OFFSET {$offset}";
$posts = $db->getAll($sql, $params);

// Get categories for filter
$categories = $db->getAll("SELECT id, name_tr FROM categories ORDER BY name_tr");

// Handle delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'])) {
    if (verifyCsrf($_POST['csrf_token'] ?? '')) {
        $db->query("DELETE FROM posts WHERE id = ?", [$_POST['delete_id']]);
        header('Location: /admin/posts.php?deleted=1');
        exit;
    }
}

include __DIR__ . '/includes/header.php';
?>

<div class="content-header">
    <h1>📝 Yazılar</h1>
    <a href="/admin/post-edit.php" class="btn btn-primary">+ Yeni Yazı</a>
</div>

<?php if (isset($_GET['deleted'])): ?>
    <div class="alert alert-success">Yazı başarıyla silindi.</div>
<?php endif; ?>

<?php if (isset($_GET['saved'])): ?>
    <div class="alert alert-success">Yazı başarıyla kaydedildi.</div>
<?php endif; ?>

<!-- Filters -->
<div class="card mb-4">
    <form method="GET" class="filters-form">
        <div class="filter-row">
            <input type="text" name="search" placeholder="Ara..." value="<?= htmlspecialchars($search) ?>" class="form-control">
            
            <select name="status" class="form-control">
                <option value="">Tüm Durumlar</option>
                <option value="published" <?= $status === 'published' ? 'selected' : '' ?>>Yayında</option>
                <option value="draft" <?= $status === 'draft' ? 'selected' : '' ?>>Taslak</option>
            </select>
            
            <select name="category" class="form-control">
                <option value="">Tüm Kategoriler</option>
                <?php foreach ($categories as $cat): ?>
                    <option value="<?= $cat['id'] ?>" <?= $category == $cat['id'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($cat['name_tr']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <button type="submit" class="btn btn-secondary">Filtrele</button>
            <a href="/admin/posts.php" class="btn btn-outline">Temizle</a>
        </div>
    </form>
</div>

<!-- Posts Table -->
<div class="card">
    <div class="table-responsive">
        <table class="data-table">
            <thead>
                <tr>
                    <th width="50">ID</th>
                    <th>Başlık</th>
                    <th>Kategori</th>
                    <th>Durum</th>
                    <th>Görüntülenme</th>
                    <th>Tarih</th>
                    <th width="150">İşlemler</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($posts)): ?>
                    <tr>
                        <td colspan="7" class="text-center">Henüz yazı yok.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($posts as $post): ?>
                        <tr>
                            <td><?= $post['id'] ?></td>
                            <td>
                                <strong><?= htmlspecialchars($post['title_tr']) ?></strong>
                                <br><small class="text-muted">/blog/<?= $post['slug'] ?></small>
                            </td>
                            <td><?= htmlspecialchars($post['category_name'] ?? '-') ?></td>
                            <td>
                                <?php if ($post['status'] === 'published'): ?>
                                    <span class="badge badge-success">Yayında</span>
                                <?php else: ?>
                                    <span class="badge badge-warning">Taslak</span>
                                <?php endif; ?>
                            </td>
                            <td><?= number_format($post['views'] ?? 0) ?></td>
                            <td><?= date('d.m.Y', strtotime($post['created_at'])) ?></td>
                            <td>
                                <div class="btn-group">
                                    <a href="/admin/post-edit.php?id=<?= $post['id'] ?>" class="btn btn-sm btn-primary">Düzenle</a>
                                    <a href="/blog/<?= $post['slug'] ?>" target="_blank" class="btn btn-sm btn-outline">Gör</a>
                                    <form method="POST" style="display:inline" onsubmit="return confirm('Bu yazıyı silmek istediğinize emin misiniz?')">
                                        <?= csrfField() ?>
                                        <input type="hidden" name="delete_id" value="<?= $post['id'] ?>">
                                        <button type="submit" class="btn btn-sm btn-danger">Sil</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?= $page - 1 ?>&status=<?= $status ?>&category=<?= $category ?>&search=<?= urlencode($search) ?>" class="btn btn-sm btn-outline">« Önceki</a>
            <?php endif; ?>
            
            <span class="pagination-info">Sayfa <?= $page ?> / <?= $totalPages ?> (Toplam: <?= $total ?>)</span>
            
            <?php if ($page < $totalPages): ?>
                <a href="?page=<?= $page + 1 ?>&status=<?= $status ?>&category=<?= $category ?>&search=<?= urlencode($search) ?>" class="btn btn-sm btn-outline">Sonraki »</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
