<?php
/**
 * Admin Panel - Post Edit
 */

require_once __DIR__ . '/../includes/config.php';

// Check authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: /admin/login.php');
    exit;
}

$id = $_GET['id'] ?? null;
$post = null;
$errors = [];

// Get categories
$categories = $db->getAll("SELECT id, name_tr FROM categories ORDER BY name_tr");

// Get tags
$allTags = $db->getAll("SELECT id, name_tr FROM tags ORDER BY name_tr");

// Load existing post
if ($id) {
    $post = $db->getRow("SELECT * FROM posts WHERE id = ?", [$id]);
    if (!$post) {
        header('Location: /admin/posts.php');
        exit;
    }
    
    // Get post tags
    $postTags = $db->getAll("SELECT tag_id FROM post_tags WHERE post_id = ?", [$id]);
    $postTagIds = array_column($postTags, 'tag_id');
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCsrf($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik hatası.';
    } else {
        $data = [
            'title_tr' => trim($_POST['title_tr'] ?? ''),
            'title_en' => trim($_POST['title_en'] ?? ''),
            'slug' => trim($_POST['slug'] ?? ''),
            'content_tr' => $_POST['content_tr'] ?? '',
            'content_en' => $_POST['content_en'] ?? '',
            'excerpt_tr' => trim($_POST['excerpt_tr'] ?? ''),
            'excerpt_en' => trim($_POST['excerpt_en'] ?? ''),
            'meta_title_tr' => trim($_POST['meta_title_tr'] ?? ''),
            'meta_title_en' => trim($_POST['meta_title_en'] ?? ''),
            'meta_description_tr' => trim($_POST['meta_description_tr'] ?? ''),
            'meta_description_en' => trim($_POST['meta_description_en'] ?? ''),
            'category_id' => $_POST['category_id'] ?: null,
            'featured_image' => trim($_POST['featured_image'] ?? ''),
            'status' => $_POST['status'] ?? 'draft',
            'is_featured' => isset($_POST['is_featured']) ? 1 : 0,
        ];
        
        // Validate
        if (empty($data['title_tr'])) {
            $errors[] = 'Başlık (TR) gerekli.';
        }
        
        // Generate slug if empty
        if (empty($data['slug'])) {
            $data['slug'] = slugify($data['title_tr']);
        }
        
        // Check slug uniqueness
        $slugCheck = $db->getRow("SELECT id FROM posts WHERE slug = ? AND id != ?", [$data['slug'], $id ?? 0]);
        if ($slugCheck) {
            $data['slug'] .= '-' . time();
        }
        
        if (empty($errors)) {
            if ($id) {
                // Update
                $data['updated_at'] = date('Y-m-d H:i:s');
                if ($data['status'] === 'published' && !$post['published_at']) {
                    $data['published_at'] = date('Y-m-d H:i:s');
                }
                
                $db->update('posts', $data, 'id = ?', [$id]);
            } else {
                // Insert
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['updated_at'] = date('Y-m-d H:i:s');
                if ($data['status'] === 'published') {
                    $data['published_at'] = date('Y-m-d H:i:s');
                }
                
                $id = $db->insert('posts', $data);
            }
            
            // Update tags
            $db->query("DELETE FROM post_tags WHERE post_id = ?", [$id]);
            $tagIds = $_POST['tags'] ?? [];
            foreach ($tagIds as $tagId) {
                $db->insert('post_tags', ['post_id' => $id, 'tag_id' => $tagId]);
            }
            
            header('Location: /admin/posts.php?saved=1');
            exit;
        }
        
        $post = $data;
        $postTagIds = $_POST['tags'] ?? [];
    }
}

include __DIR__ . '/includes/header.php';
?>

<div class="content-header">
    <h1><?= $id ? '✏️ Yazı Düzenle' : '➕ Yeni Yazı' ?></h1>
    <a href="/admin/posts.php" class="btn btn-outline">← Geri</a>
</div>

<?php if ($errors): ?>
    <div class="alert alert-danger">
        <?php foreach ($errors as $error): ?>
            <p><?= htmlspecialchars($error) ?></p>
        <?php endforeach; ?>
    </div>
<?php endif; ?>

<form method="POST" class="post-form">
    <?= csrfField() ?>
    
    <div class="form-grid">
        <!-- Main Content -->
        <div class="form-main">
            <div class="card">
                <div class="card-header">
                    <h3>İçerik (Türkçe)</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label>Başlık *</label>
                        <input type="text" name="title_tr" value="<?= htmlspecialchars($post['title_tr'] ?? '') ?>" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label>URL Slug</label>
                        <input type="text" name="slug" value="<?= htmlspecialchars($post['slug'] ?? '') ?>" class="form-control" placeholder="otomatik-olusturulur">
                    </div>
                    
                    <div class="form-group">
                        <label>Özet</label>
                        <textarea name="excerpt_tr" rows="3" class="form-control"><?= htmlspecialchars($post['excerpt_tr'] ?? '') ?></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>İçerik</label>
                        <textarea name="content_tr" rows="15" class="form-control editor"><?= htmlspecialchars($post['content_tr'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>İçerik (English)</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label>Title</label>
                        <input type="text" name="title_en" value="<?= htmlspecialchars($post['title_en'] ?? '') ?>" class="form-control">
                    </div>
                    
                    <div class="form-group">
                        <label>Excerpt</label>
                        <textarea name="excerpt_en" rows="3" class="form-control"><?= htmlspecialchars($post['excerpt_en'] ?? '') ?></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>Content</label>
                        <textarea name="content_en" rows="15" class="form-control editor"><?= htmlspecialchars($post['content_en'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>SEO Ayarları</h3>
                </div>
                <div class="card-body">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Meta Title (TR)</label>
                            <input type="text" name="meta_title_tr" value="<?= htmlspecialchars($post['meta_title_tr'] ?? '') ?>" class="form-control" maxlength="70">
                            <small>Max 70 karakter</small>
                        </div>
                        <div class="form-group">
                            <label>Meta Title (EN)</label>
                            <input type="text" name="meta_title_en" value="<?= htmlspecialchars($post['meta_title_en'] ?? '') ?>" class="form-control" maxlength="70">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Meta Description (TR)</label>
                            <textarea name="meta_description_tr" rows="2" class="form-control" maxlength="160"><?= htmlspecialchars($post['meta_description_tr'] ?? '') ?></textarea>
                            <small>Max 160 karakter</small>
                        </div>
                        <div class="form-group">
                            <label>Meta Description (EN)</label>
                            <textarea name="meta_description_en" rows="2" class="form-control" maxlength="160"><?= htmlspecialchars($post['meta_description_en'] ?? '') ?></textarea>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Sidebar -->
        <div class="form-sidebar">
            <div class="card">
                <div class="card-header">
                    <h3>Yayın</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label>Durum</label>
                        <select name="status" class="form-control">
                            <option value="draft" <?= ($post['status'] ?? '') === 'draft' ? 'selected' : '' ?>>Taslak</option>
                            <option value="published" <?= ($post['status'] ?? '') === 'published' ? 'selected' : '' ?>>Yayında</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" name="is_featured" value="1" <?= ($post['is_featured'] ?? 0) ? 'checked' : '' ?>>
                            Öne Çıkan Yazı
                        </label>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary btn-block">Kaydet</button>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Kategori</h3>
                </div>
                <div class="card-body">
                    <select name="category_id" class="form-control">
                        <option value="">Kategori Seç</option>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?= $cat['id'] ?>" <?= ($post['category_id'] ?? '') == $cat['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($cat['name_tr']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Etiketler</h3>
                </div>
                <div class="card-body">
                    <div class="tags-list">
                        <?php foreach ($allTags as $tag): ?>
                            <label class="tag-checkbox">
                                <input type="checkbox" name="tags[]" value="<?= $tag['id'] ?>" 
                                       <?= in_array($tag['id'], $postTagIds ?? []) ? 'checked' : '' ?>>
                                <?= htmlspecialchars($tag['name_tr']) ?>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Öne Çıkan Görsel</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <input type="text" name="featured_image" value="<?= htmlspecialchars($post['featured_image'] ?? '') ?>" class="form-control" placeholder="/uploads/images/...">
                    </div>
                    <?php if (!empty($post['featured_image'])): ?>
                        <img src="<?= htmlspecialchars($post['featured_image']) ?>" alt="Preview" class="image-preview">
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</form>

<style>
.form-grid {
    display: grid;
    grid-template-columns: 1fr 320px;
    gap: 24px;
}
.form-main .card { margin-bottom: 24px; }
.form-sidebar .card { margin-bottom: 16px; }
.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 16px;
}
.form-group { margin-bottom: 16px; }
.form-group label { display: block; margin-bottom: 6px; font-weight: 500; }
.form-group small { color: #64748b; font-size: 12px; }
.card-header { padding: 16px; border-bottom: 1px solid #e2e8f0; }
.card-header h3 { margin: 0; font-size: 16px; }
.card-body { padding: 16px; }
.tags-list { max-height: 200px; overflow-y: auto; }
.tag-checkbox { display: block; padding: 4px 0; cursor: pointer; }
.tag-checkbox input { margin-right: 8px; }
.checkbox-label { display: flex; align-items: center; cursor: pointer; }
.checkbox-label input { margin-right: 8px; }
.image-preview { max-width: 100%; margin-top: 8px; border-radius: 8px; }
.btn-block { width: 100%; }
.editor { font-family: monospace; }

@media (max-width: 1024px) {
    .form-grid { grid-template-columns: 1fr; }
    .form-row { grid-template-columns: 1fr; }
}
</style>

<?php include __DIR__ . '/includes/footer.php'; ?>
