<?php
/**
 * Admin Panel - Page Edit
 */

require_once __DIR__ . '/../includes/config.php';

// Check authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: /admin/login.php');
    exit;
}

$id = $_GET['id'] ?? null;
$pageData = null;
$errors = [];

// Load existing page
if ($id) {
    $pageData = $db->getRow("SELECT * FROM pages WHERE id = ?", [$id]);
    if (!$pageData) {
        header('Location: /admin/pages.php');
        exit;
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCsrf($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik hatası.';
    } else {
        $data = [
            'title_tr' => trim($_POST['title_tr'] ?? ''),
            'title_en' => trim($_POST['title_en'] ?? ''),
            'slug' => trim($_POST['slug'] ?? ''),
            'content_tr' => $_POST['content_tr'] ?? '',
            'content_en' => $_POST['content_en'] ?? '',
            'excerpt_tr' => trim($_POST['excerpt_tr'] ?? ''),
            'meta_title_tr' => trim($_POST['meta_title_tr'] ?? ''),
            'meta_title_en' => trim($_POST['meta_title_en'] ?? ''),
            'meta_description_tr' => trim($_POST['meta_description_tr'] ?? ''),
            'meta_description_en' => trim($_POST['meta_description_en'] ?? ''),
            'featured_image' => trim($_POST['featured_image'] ?? ''),
            'template' => $_POST['template'] ?? 'default',
            'sort_order' => intval($_POST['sort_order'] ?? 0),
            'status' => $_POST['status'] ?? 'draft',
        ];
        
        // Validate
        if (empty($data['title_tr'])) {
            $errors[] = 'Başlık (TR) gerekli.';
        }
        
        // Generate slug if empty
        if (empty($data['slug'])) {
            $data['slug'] = slugify($data['title_tr']);
        }
        
        // Check slug uniqueness
        $slugCheck = $db->getRow("SELECT id FROM pages WHERE slug = ? AND id != ?", [$data['slug'], $id ?? 0]);
        if ($slugCheck) {
            $data['slug'] .= '-' . time();
        }
        
        if (empty($errors)) {
            if ($id) {
                // Update
                $data['updated_at'] = date('Y-m-d H:i:s');
                if ($data['status'] === 'published' && !$pageData['published_at']) {
                    $data['published_at'] = date('Y-m-d H:i:s');
                }
                
                $db->update('pages', $data, 'id = ?', [$id]);
            } else {
                // Insert
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['updated_at'] = date('Y-m-d H:i:s');
                if ($data['status'] === 'published') {
                    $data['published_at'] = date('Y-m-d H:i:s');
                }
                
                $id = $db->insert('pages', $data);
            }
            
            header('Location: /admin/pages.php?saved=1');
            exit;
        }
        
        $pageData = $data;
    }
}

include __DIR__ . '/includes/header.php';
?>

<div class="content-header">
    <h1><?= $id ? '✏️ Sayfa Düzenle' : '➕ Yeni Sayfa' ?></h1>
    <a href="/admin/pages.php" class="btn btn-outline">← Geri</a>
</div>

<?php if ($errors): ?>
    <div class="alert alert-danger">
        <?php foreach ($errors as $error): ?>
            <p><?= htmlspecialchars($error) ?></p>
        <?php endforeach; ?>
    </div>
<?php endif; ?>

<form method="POST" class="post-form">
    <?= csrfField() ?>
    
    <div class="form-grid">
        <!-- Main Content -->
        <div class="form-main">
            <div class="card">
                <div class="card-header">
                    <h3>İçerik (Türkçe)</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label>Başlık *</label>
                        <input type="text" name="title_tr" value="<?= htmlspecialchars($pageData['title_tr'] ?? '') ?>" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label>URL Slug</label>
                        <input type="text" name="slug" value="<?= htmlspecialchars($pageData['slug'] ?? '') ?>" class="form-control" placeholder="otomatik-olusturulur">
                    </div>
                    
                    <div class="form-group">
                        <label>Özet</label>
                        <textarea name="excerpt_tr" rows="3" class="form-control"><?= htmlspecialchars($pageData['excerpt_tr'] ?? '') ?></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>İçerik</label>
                        <textarea name="content_tr" rows="20" class="form-control editor"><?= htmlspecialchars($pageData['content_tr'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>İçerik (English)</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label>Title</label>
                        <input type="text" name="title_en" value="<?= htmlspecialchars($pageData['title_en'] ?? '') ?>" class="form-control">
                    </div>
                    
                    <div class="form-group">
                        <label>Content</label>
                        <textarea name="content_en" rows="20" class="form-control editor"><?= htmlspecialchars($pageData['content_en'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>SEO Ayarları</h3>
                </div>
                <div class="card-body">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Meta Title (TR)</label>
                            <input type="text" name="meta_title_tr" value="<?= htmlspecialchars($pageData['meta_title_tr'] ?? '') ?>" class="form-control" maxlength="70">
                        </div>
                        <div class="form-group">
                            <label>Meta Title (EN)</label>
                            <input type="text" name="meta_title_en" value="<?= htmlspecialchars($pageData['meta_title_en'] ?? '') ?>" class="form-control" maxlength="70">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Meta Description (TR)</label>
                            <textarea name="meta_description_tr" rows="2" class="form-control" maxlength="160"><?= htmlspecialchars($pageData['meta_description_tr'] ?? '') ?></textarea>
                        </div>
                        <div class="form-group">
                            <label>Meta Description (EN)</label>
                            <textarea name="meta_description_en" rows="2" class="form-control" maxlength="160"><?= htmlspecialchars($pageData['meta_description_en'] ?? '') ?></textarea>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Sidebar -->
        <div class="form-sidebar">
            <div class="card">
                <div class="card-header">
                    <h3>Yayın</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label>Durum</label>
                        <select name="status" class="form-control">
                            <option value="draft" <?= ($pageData['status'] ?? '') === 'draft' ? 'selected' : '' ?>>Taslak</option>
                            <option value="published" <?= ($pageData['status'] ?? '') === 'published' ? 'selected' : '' ?>>Yayında</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Sıralama</label>
                        <input type="number" name="sort_order" value="<?= $pageData['sort_order'] ?? 0 ?>" class="form-control">
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary btn-block">Kaydet</button>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Şablon</h3>
                </div>
                <div class="card-body">
                    <select name="template" class="form-control">
                        <option value="default" <?= ($pageData['template'] ?? '') === 'default' ? 'selected' : '' ?>>Varsayılan</option>
                        <option value="contact" <?= ($pageData['template'] ?? '') === 'contact' ? 'selected' : '' ?>>İletişim</option>
                        <option value="about" <?= ($pageData['template'] ?? '') === 'about' ? 'selected' : '' ?>>Hakkımda</option>
                        <option value="full-width" <?= ($pageData['template'] ?? '') === 'full-width' ? 'selected' : '' ?>>Tam Genişlik</option>
                    </select>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Öne Çıkan Görsel</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <input type="text" name="featured_image" value="<?= htmlspecialchars($pageData['featured_image'] ?? '') ?>" class="form-control" placeholder="/uploads/images/...">
                    </div>
                    <?php if (!empty($pageData['featured_image'])): ?>
                        <img src="<?= htmlspecialchars($pageData['featured_image']) ?>" alt="Preview" class="image-preview">
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</form>

<style>
.form-grid { display: grid; grid-template-columns: 1fr 320px; gap: 24px; }
.form-main .card { margin-bottom: 24px; }
.form-sidebar .card { margin-bottom: 16px; }
.form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; }
.form-group { margin-bottom: 16px; }
.form-group label { display: block; margin-bottom: 6px; font-weight: 500; }
.card-header { padding: 16px; border-bottom: 1px solid #e2e8f0; }
.card-header h3 { margin: 0; font-size: 16px; }
.card-body { padding: 16px; }
.image-preview { max-width: 100%; margin-top: 8px; border-radius: 8px; }
.btn-block { width: 100%; }
.editor { font-family: monospace; }
@media (max-width: 1024px) { .form-grid { grid-template-columns: 1fr; } .form-row { grid-template-columns: 1fr; } }
</style>

<?php include __DIR__ . '/includes/footer.php'; ?>
