<?php
/**
 * Admin Panel - Media Management
 */

require_once __DIR__ . '/../includes/config.php';

// Check authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: /admin/login.php');
    exit;
}

$message = '';
$error = '';

// Handle upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    if (!verifyCsrf($_POST['csrf_token'] ?? '')) {
        $error = 'Güvenlik hatası.';
    } else {
        $file = $_FILES['file'];
        
        if ($file['error'] === UPLOAD_ERR_OK) {
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'application/pdf'];
            $maxSize = 10 * 1024 * 1024; // 10MB
            
            if (!in_array($file['type'], $allowedTypes)) {
                $error = 'Geçersiz dosya türü. İzin verilen: JPG, PNG, GIF, WebP, PDF';
            } elseif ($file['size'] > $maxSize) {
                $error = 'Dosya çok büyük. Maksimum 10MB.';
            } else {
                // Generate unique filename
                $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = date('Y-m') . '/' . uniqid() . '-' . slugify(pathinfo($file['name'], PATHINFO_FILENAME)) . '.' . $ext;
                $uploadPath = UPLOADS_PATH . '/images/' . $filename;
                
                // Create directory if needed
                $dir = dirname($uploadPath);
                if (!is_dir($dir)) {
                    mkdir($dir, 0755, true);
                }
                
                if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
                    // Save to database
                    $db->insert('media', [
                        'filename' => $filename,
                        'original_name' => $file['name'],
                        'mime_type' => $file['type'],
                        'file_size' => $file['size'],
                        'file_path' => '/uploads/images/' . $filename,
                        'uploaded_by' => $_SESSION['admin_id'],
                        'created_at' => date('Y-m-d H:i:s')
                    ]);
                    $message = 'Dosya yüklendi.';
                } else {
                    $error = 'Dosya yüklenemedi.';
                }
            }
        } else {
            $error = 'Dosya yükleme hatası.';
        }
    }
}

// Handle delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'])) {
    if (verifyCsrf($_POST['csrf_token'] ?? '')) {
        $media = $db->getRow("SELECT * FROM media WHERE id = ?", [$_POST['delete_id']]);
        if ($media) {
            $filePath = UPLOADS_PATH . '/images/' . $media['filename'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            $db->query("DELETE FROM media WHERE id = ?", [$media['id']]);
            $message = 'Dosya silindi.';
        }
    }
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 24;
$offset = ($page - 1) * $perPage;

$total = $db->getValue("SELECT COUNT(*) FROM media");
$totalPages = ceil($total / $perPage);

$mediaList = $db->getAll("SELECT * FROM media ORDER BY created_at DESC LIMIT {$perPage} OFFSET {$offset}");

include __DIR__ . '/includes/header.php';
?>

<div class="content-header">
    <h1>🖼️ Medya</h1>
    <button onclick="document.getElementById('uploadModal').style.display='flex'" class="btn btn-primary">+ Dosya Yükle</button>
</div>

<?php if ($message): ?>
    <div class="alert alert-success"><?= htmlspecialchars($message) ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
<?php endif; ?>

<div class="card">
    <div class="media-grid">
        <?php if (empty($mediaList)): ?>
            <p class="text-center" style="grid-column: 1/-1; padding: 40px;">Henüz medya yok.</p>
        <?php else: ?>
            <?php foreach ($mediaList as $media): ?>
                <div class="media-item">
                    <?php if (strpos($media['mime_type'], 'image/') === 0): ?>
                        <img src="<?= htmlspecialchars($media['file_path']) ?>" alt="">
                    <?php else: ?>
                        <div class="media-icon">📄</div>
                    <?php endif; ?>
                    <div class="media-info">
                        <span class="media-name" title="<?= htmlspecialchars($media['original_name']) ?>">
                            <?= htmlspecialchars(truncate($media['original_name'], 20)) ?>
                        </span>
                        <span class="media-size"><?= formatBytes($media['file_size']) ?></span>
                    </div>
                    <div class="media-actions">
                        <button onclick="copyToClipboard('<?= $media['file_path'] ?>')" class="btn btn-sm btn-outline" title="URL Kopyala">📋</button>
                        <form method="POST" style="display:inline" onsubmit="return confirm('Silmek istediğinize emin misiniz?')">
                            <?= csrfField() ?>
                            <input type="hidden" name="delete_id" value="<?= $media['id'] ?>">
                            <button type="submit" class="btn btn-sm btn-danger" title="Sil">🗑️</button>
                        </form>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    
    <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?= $page - 1 ?>" class="btn btn-sm btn-outline">« Önceki</a>
            <?php endif; ?>
            <span class="pagination-info">Sayfa <?= $page ?> / <?= $totalPages ?></span>
            <?php if ($page < $totalPages): ?>
                <a href="?page=<?= $page + 1 ?>" class="btn btn-sm btn-outline">Sonraki »</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<!-- Upload Modal -->
<div id="uploadModal" class="modal" style="display:none">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Dosya Yükle</h3>
            <button onclick="document.getElementById('uploadModal').style.display='none'" class="modal-close">&times;</button>
        </div>
        <form method="POST" enctype="multipart/form-data">
            <?= csrfField() ?>
            <div class="modal-body">
                <div class="upload-zone" id="dropZone">
                    <input type="file" name="file" id="fileInput" accept="image/*,application/pdf" required>
                    <div class="upload-text">
                        <span>📁</span>
                        <p>Dosya seçin veya sürükleyin</p>
                        <small>JPG, PNG, GIF, WebP, PDF • Max 10MB</small>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" onclick="document.getElementById('uploadModal').style.display='none'" class="btn btn-outline">İptal</button>
                <button type="submit" class="btn btn-primary">Yükle</button>
            </div>
        </form>
    </div>
</div>

<style>
.media-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
    gap: 16px;
    padding: 16px;
}
.media-item {
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    overflow: hidden;
    background: #f8fafc;
}
.media-item img {
    width: 100%;
    height: 140px;
    object-fit: cover;
}
.media-icon {
    width: 100%;
    height: 140px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 48px;
    background: #e2e8f0;
}
.media-info {
    padding: 8px 12px;
}
.media-name {
    display: block;
    font-size: 13px;
    font-weight: 500;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}
.media-size {
    font-size: 12px;
    color: #64748b;
}
.media-actions {
    padding: 8px 12px;
    border-top: 1px solid #e2e8f0;
    display: flex;
    gap: 8px;
}
.modal {
    position: fixed; top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(0,0,0,0.5); display: flex; align-items: center; justify-content: center; z-index: 1000;
}
.modal-content { background: white; border-radius: 12px; width: 100%; max-width: 500px; }
.modal-header { padding: 20px; border-bottom: 1px solid #e2e8f0; display: flex; justify-content: space-between; align-items: center; }
.modal-header h3 { margin: 0; }
.modal-close { background: none; border: none; font-size: 24px; cursor: pointer; color: #64748b; }
.modal-body { padding: 20px; }
.modal-footer { padding: 20px; border-top: 1px solid #e2e8f0; display: flex; gap: 12px; justify-content: flex-end; }
.upload-zone {
    border: 2px dashed #cbd5e1;
    border-radius: 8px;
    padding: 40px;
    text-align: center;
    cursor: pointer;
    transition: border-color 0.2s;
}
.upload-zone:hover { border-color: #0ea5e9; }
.upload-zone input { display: none; }
.upload-text span { font-size: 48px; display: block; margin-bottom: 8px; }
.upload-text p { margin: 0; font-weight: 500; }
.upload-text small { color: #64748b; }
</style>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        alert('URL kopyalandı: ' + text);
    });
}

document.getElementById('dropZone').addEventListener('click', () => {
    document.getElementById('fileInput').click();
});

document.getElementById('uploadModal').addEventListener('click', function(e) {
    if (e.target === this) this.style.display = 'none';
});
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

<?php
function formatBytes($bytes) {
    if ($bytes < 1024) return $bytes . ' B';
    if ($bytes < 1048576) return round($bytes / 1024, 1) . ' KB';
    return round($bytes / 1048576, 1) . ' MB';
}
?>
