<?php
/**
 * Admin Panel - Categories Management
 */

require_once __DIR__ . '/../includes/config.php';

// Check authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: /admin/login.php');
    exit;
}

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && verifyCsrf($_POST['csrf_token'] ?? '')) {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        $name = trim($_POST['name_tr'] ?? '');
        $slug = trim($_POST['slug'] ?? '') ?: slugify($name);
        $description = trim($_POST['description_tr'] ?? '');
        $parentId = $_POST['parent_id'] ?: null;
        
        if ($name) {
            $db->insert('categories', [
                'name_tr' => $name,
                'name_en' => $name,
                'slug' => $slug,
                'description_tr' => $description,
                'parent_id' => $parentId
            ]);
            $message = 'Kategori eklendi.';
        }
    }
    
    if ($action === 'edit') {
        $id = $_POST['id'];
        $name = trim($_POST['name_tr'] ?? '');
        $slug = trim($_POST['slug'] ?? '');
        $description = trim($_POST['description_tr'] ?? '');
        $parentId = $_POST['parent_id'] ?: null;
        
        if ($name && $id) {
            $db->update('categories', [
                'name_tr' => $name,
                'slug' => $slug,
                'description_tr' => $description,
                'parent_id' => $parentId
            ], 'id = ?', [$id]);
            $message = 'Kategori güncellendi.';
        }
    }
    
    if ($action === 'delete') {
        $id = $_POST['id'];
        // Check if has posts
        $postCount = $db->getValue("SELECT COUNT(*) FROM posts WHERE category_id = ?", [$id]);
        if ($postCount > 0) {
            $error = 'Bu kategoride yazılar var, silinemez.';
        } else {
            $db->query("DELETE FROM categories WHERE id = ?", [$id]);
            $message = 'Kategori silindi.';
        }
    }
}

// Get categories
$categories = $db->getAll("
    SELECT c.*, 
           (SELECT COUNT(*) FROM posts WHERE category_id = c.id) as post_count,
           p.name_tr as parent_name
    FROM categories c
    LEFT JOIN categories p ON c.parent_id = p.id
    ORDER BY c.name_tr
");

include __DIR__ . '/includes/header.php';
?>

<div class="content-header">
    <h1>📁 Kategoriler</h1>
    <button onclick="showAddModal()" class="btn btn-primary">+ Yeni Kategori</button>
</div>

<?php if ($message): ?>
    <div class="alert alert-success"><?= htmlspecialchars($message) ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
<?php endif; ?>

<div class="card">
    <div class="table-responsive">
        <table class="data-table">
            <thead>
                <tr>
                    <th width="50">ID</th>
                    <th>Kategori Adı</th>
                    <th>Slug</th>
                    <th>Üst Kategori</th>
                    <th>Yazı Sayısı</th>
                    <th width="150">İşlemler</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($categories)): ?>
                    <tr>
                        <td colspan="6" class="text-center">Henüz kategori yok.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($categories as $cat): ?>
                        <tr>
                            <td><?= $cat['id'] ?></td>
                            <td><strong><?= htmlspecialchars($cat['name_tr']) ?></strong></td>
                            <td><code><?= htmlspecialchars($cat['slug']) ?></code></td>
                            <td><?= htmlspecialchars($cat['parent_name'] ?? '-') ?></td>
                            <td><?= $cat['post_count'] ?></td>
                            <td>
                                <button onclick="showEditModal(<?= htmlspecialchars(json_encode($cat)) ?>)" class="btn btn-sm btn-primary">Düzenle</button>
                                <form method="POST" style="display:inline" onsubmit="return confirm('Bu kategoriyi silmek istediğinize emin misiniz?')">
                                    <?= csrfField() ?>
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="id" value="<?= $cat['id'] ?>">
                                    <button type="submit" class="btn btn-sm btn-danger">Sil</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add/Edit Modal -->
<div id="categoryModal" class="modal" style="display:none">
    <div class="modal-content">
        <div class="modal-header">
            <h3 id="modalTitle">Kategori Ekle</h3>
            <button onclick="closeModal()" class="modal-close">&times;</button>
        </div>
        <form method="POST">
            <?= csrfField() ?>
            <input type="hidden" name="action" id="formAction" value="add">
            <input type="hidden" name="id" id="formId" value="">
            
            <div class="modal-body">
                <div class="form-group">
                    <label>Kategori Adı *</label>
                    <input type="text" name="name_tr" id="formName" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label>URL Slug</label>
                    <input type="text" name="slug" id="formSlug" class="form-control" placeholder="otomatik">
                </div>
                
                <div class="form-group">
                    <label>Üst Kategori</label>
                    <select name="parent_id" id="formParent" class="form-control">
                        <option value="">Yok (Ana Kategori)</option>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?= $cat['id'] ?>"><?= htmlspecialchars($cat['name_tr']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>Açıklama</label>
                    <textarea name="description_tr" id="formDesc" rows="3" class="form-control"></textarea>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" onclick="closeModal()" class="btn btn-outline">İptal</button>
                <button type="submit" class="btn btn-primary">Kaydet</button>
            </div>
        </form>
    </div>
</div>

<style>
.modal {
    position: fixed; top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(0,0,0,0.5); display: flex; align-items: center; justify-content: center; z-index: 1000;
}
.modal-content {
    background: white; border-radius: 12px; width: 100%; max-width: 500px; max-height: 90vh; overflow: auto;
}
.modal-header {
    padding: 20px; border-bottom: 1px solid #e2e8f0; display: flex; justify-content: space-between; align-items: center;
}
.modal-header h3 { margin: 0; }
.modal-close { background: none; border: none; font-size: 24px; cursor: pointer; color: #64748b; }
.modal-body { padding: 20px; }
.modal-footer { padding: 20px; border-top: 1px solid #e2e8f0; display: flex; gap: 12px; justify-content: flex-end; }
.form-group { margin-bottom: 16px; }
.form-group label { display: block; margin-bottom: 6px; font-weight: 500; }
</style>

<script>
function showAddModal() {
    document.getElementById('modalTitle').textContent = 'Kategori Ekle';
    document.getElementById('formAction').value = 'add';
    document.getElementById('formId').value = '';
    document.getElementById('formName').value = '';
    document.getElementById('formSlug').value = '';
    document.getElementById('formParent').value = '';
    document.getElementById('formDesc').value = '';
    document.getElementById('categoryModal').style.display = 'flex';
}

function showEditModal(cat) {
    document.getElementById('modalTitle').textContent = 'Kategori Düzenle';
    document.getElementById('formAction').value = 'edit';
    document.getElementById('formId').value = cat.id;
    document.getElementById('formName').value = cat.name_tr;
    document.getElementById('formSlug').value = cat.slug;
    document.getElementById('formParent').value = cat.parent_id || '';
    document.getElementById('formDesc').value = cat.description_tr || '';
    document.getElementById('categoryModal').style.display = 'flex';
}

function closeModal() {
    document.getElementById('categoryModal').style.display = 'none';
}

document.getElementById('categoryModal').addEventListener('click', function(e) {
    if (e.target === this) closeModal();
});
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
