<?php
/**
 * Dr. Özgür Oktay Nar - PHP CMS
 * Front Controller - Ana Giriş Noktası
 */

// Define root path (önce tanımlı mı kontrol et)
if (!defined('ROOT_PATH')) {
    define('ROOT_PATH', dirname(__DIR__));
}

// Load configuration
require_once ROOT_PATH . '/includes/config.php';

// View class'ını yükle
require_once ROOT_PATH . '/includes/View.php';
require_once ROOT_PATH . '/includes/Router.php';

// Session zaten config.php'de başlatıldı

// Initialize database
$db = Database::getInstance();

// Initialize View
View::init('cliniq');

// Get current language from URL or session
$uri = $_SERVER['REQUEST_URI'];
$lang = DEFAULT_LANG;

// Check for language prefix in URL (e.g., /en/about)
if (preg_match('#^/(en|tr)(/.*)?$#', $uri, $matches)) {
    $lang = $matches[1];
    $_SESSION['lang'] = $lang;
}

// Set global language
define('CURRENT_LANG', $lang);

// Initialize Router
$router = new Router();

// =====================================================
// ROUTES
// =====================================================

// Homepage
$router->get('/', function() {
    $controller = new HomeController();
    $controller->index();
});

// English homepage
$router->get('/en', function() {
    $controller = new HomeController();
    $controller->index();
});

// About page
$router->get('/hakkimda', function() {
    $controller = new PageController();
    $controller->show('hakkimda');
});

$router->get('/en/about', function() {
    $controller = new PageController();
    $controller->show('hakkimda');
});

// Services
$router->get('/uzmanlik-alanlari', function() {
    $controller = new ServiceController();
    $controller->index();
});

$router->get('/uzmanlik-alanlari/{slug}', function($slug) {
    $controller = new ServiceController();
    $controller->show($slug);
});

// Blog
$router->get('/blog', function() {
    $controller = new BlogController();
    $controller->index();
});

$router->get('/en/blog', function() {
    $controller = new BlogController();
    $controller->index();
});

$router->get('/blog/{slug}', function($slug) {
    $controller = new BlogController();
    $controller->show($slug);
});

$router->get('/en/blog/{slug}', function($slug) {
    $controller = new BlogController();
    $controller->show($slug);
});

// Blog category
$router->get('/kategori/{slug}', function($slug) {
    $controller = new BlogController();
    $controller->category($slug);
});

// Blog tag
$router->get('/etiket/{slug}', function($slug) {
    $controller = new BlogController();
    $controller->tag($slug);
});

// Contact
$router->get('/iletisim', function() {
    $controller = new PageController();
    $controller->show('iletisim');
});

$router->get('/en/contact', function() {
    $controller = new PageController();
    $controller->show('iletisim');
});

// Search
$router->get('/ara', function() {
    $controller = new SearchController();
    $controller->index();
});

// Sitemap
$router->get('/sitemap.xml', function() {
    $controller = new SitemapController();
    $controller->xml();
});

// Robots.txt
$router->get('/robots.txt', function() {
    header('Content-Type: text/plain');
    echo getSetting('robots_txt', "User-agent: *\nAllow: /");
});

// Generic page handler (fallback)
$router->get('/{slug}', function($slug) {
    $controller = new PageController();
    $controller->show($slug);
});

$router->get('/en/{slug}', function($slug) {
    $controller = new PageController();
    $controller->show($slug);
});

// 404 handler
$router->notFound(function() {
    View::set('pageTitle', '404 - Sayfa Bulunamadı');
    View::render('404');
});

// =====================================================
// CONTROLLERS (Inline for simplicity - can be moved to separate files)
// =====================================================

class HomeController {
    public function index() {
        global $db;
        
        // Get featured posts
        $featuredPosts = $db->getAll("
            SELECT * FROM posts 
            WHERE status = 'published' AND is_featured = 1 
            ORDER BY published_at DESC LIMIT 3
        ");
        
        // Get latest posts
        $latestPosts = $db->getAll("
            SELECT * FROM posts 
            WHERE status = 'published' 
            ORDER BY published_at DESC LIMIT 6
        ");
        
        // Get services
        $services = $db->getAll("
            SELECT * FROM services 
            WHERE status = 'published' AND show_on_home = 1 
            ORDER BY sort_order LIMIT 6
        ");
        
        // Get FAQs
        $faqs = $db->getAll("
            SELECT * FROM faqs 
            WHERE status = 'active' AND show_on_home = 1 
            ORDER BY sort_order LIMIT 6
        ");
        
        // Get videos
        $videos = $db->getAll("
            SELECT * FROM videos 
            WHERE status = 'active' AND show_on_home = 1 
            ORDER BY sort_order LIMIT 4
        ");
        
        // Get reviews (if enabled)
        $reviews = [];
        if (FEATURE_GOOGLE_REVIEWS) {
            $reviews = $db->getAll("
                SELECT * FROM reviews 
                WHERE show_on_site = 1 
                ORDER BY is_featured DESC, review_time DESC LIMIT 6
            ");
        }
        
        View::set('pageTitle', getSetting('site_name_' . CURRENT_LANG));
        View::set('metaDescription', getSetting('site_description_' . CURRENT_LANG));
        View::set('featuredPosts', $featuredPosts);
        View::set('latestPosts', $latestPosts);
        View::set('services', $services);
        View::set('faqs', $faqs);
        View::set('videos', $videos);
        View::set('reviews', $reviews);
        
        View::render('home');
    }
}

class BlogController {
    public function index() {
        global $db;
        
        $page = isset($_GET['sayfa']) ? max(1, (int)$_GET['sayfa']) : 1;
        $perPage = 12;
        $offset = ($page - 1) * $perPage;
        
        $total = $db->getValue("SELECT COUNT(*) FROM posts WHERE status = 'published'");
        $totalPages = ceil($total / $perPage);
        
        $posts = $db->getAll("
            SELECT p.*, c.name_" . CURRENT_LANG . " as category_name, c.slug as category_slug
            FROM posts p
            LEFT JOIN categories c ON p.category_id = c.id
            WHERE p.status = 'published'
            ORDER BY p.published_at DESC
            LIMIT {$perPage} OFFSET {$offset}
        ");
        
        $categories = $db->getAll("
            SELECT *, name_" . CURRENT_LANG . " as name 
            FROM categories 
            WHERE status = 'active' AND post_count > 0 
            ORDER BY name_" . CURRENT_LANG
        );
        
        View::set('pageTitle', CURRENT_LANG === 'tr' ? 'Blog' : 'Blog');
        View::set('posts', $posts);
        View::set('categories', $categories);
        View::set('currentPage', $page);
        View::set('totalPages', $totalPages);
        
        View::render('blog');
    }
    
    public function show($slug) {
        global $db;
        
        $post = $db->getRow("
            SELECT p.*, c.name_" . CURRENT_LANG . " as category_name, c.slug as category_slug
            FROM posts p
            LEFT JOIN categories c ON p.category_id = c.id
            WHERE p.slug = ? AND p.status = 'published'
        ", [$slug]);
        
        if (!$post) {
            http_response_code(404);
            View::render('404');
            return;
        }
        
        // Increment views
        $db->query("UPDATE posts SET views = views + 1 WHERE id = ?", [$post['id']]);
        
        // Get related posts
        $relatedPosts = $db->getAll("
            SELECT * FROM posts 
            WHERE status = 'published' 
              AND id != ? 
              AND category_id = ?
            ORDER BY published_at DESC LIMIT 3
        ", [$post['id'], $post['category_id']]);
        
        // Get post tags
        $tags = $db->getAll("
            SELECT t.* FROM tags t
            INNER JOIN post_tags pt ON t.id = pt.tag_id
            WHERE pt.post_id = ?
        ", [$post['id']]);
        
        View::set('pageTitle', localized($post, 'meta_title') ?: localized($post, 'title'));
        View::set('metaDescription', localized($post, 'meta_description'));
        View::set('post', $post);
        View::set('relatedPosts', $relatedPosts);
        View::set('tags', $tags);
        
        View::render('blog-single');
    }
    
    public function category($slug) {
        global $db;
        
        $category = $db->getRow("SELECT * FROM categories WHERE slug = ?", [$slug]);
        
        if (!$category) {
            http_response_code(404);
            View::render('404');
            return;
        }
        
        $page = isset($_GET['sayfa']) ? max(1, (int)$_GET['sayfa']) : 1;
        $perPage = 12;
        $offset = ($page - 1) * $perPage;
        
        $total = $db->getValue("SELECT COUNT(*) FROM posts WHERE status = 'published' AND category_id = ?", [$category['id']]);
        $totalPages = ceil($total / $perPage);
        
        $posts = $db->getAll("
            SELECT * FROM posts 
            WHERE status = 'published' AND category_id = ?
            ORDER BY published_at DESC
            LIMIT {$perPage} OFFSET {$offset}
        ", [$category['id']]);
        
        View::set('pageTitle', localized($category, 'name'));
        View::set('category', $category);
        View::set('posts', $posts);
        View::set('currentPage', $page);
        View::set('totalPages', $totalPages);
        
        View::render('blog-category');
    }
    
    public function tag($slug) {
        global $db;
        
        $tag = $db->getRow("SELECT * FROM tags WHERE slug = ?", [$slug]);
        
        if (!$tag) {
            http_response_code(404);
            View::render('404');
            return;
        }
        
        $posts = $db->getAll("
            SELECT p.* FROM posts p
            INNER JOIN post_tags pt ON p.id = pt.post_id
            WHERE pt.tag_id = ? AND p.status = 'published'
            ORDER BY p.published_at DESC
        ", [$tag['id']]);
        
        View::set('pageTitle', localized($tag, 'name'));
        View::set('tag', $tag);
        View::set('posts', $posts);
        
        View::render('blog-tag');
    }
}

class PageController {
    public function show($slug) {
        global $db;
        
        $page = $db->getRow("
            SELECT * FROM pages 
            WHERE slug = ? AND status = 'published'
        ", [$slug]);
        
        if (!$page) {
            http_response_code(404);
            View::render('404');
            return;
        }
        
        // Increment views
        $db->query("UPDATE pages SET views = views + 1 WHERE id = ?", [$page['id']]);
        
        View::set('pageTitle', localized($page, 'meta_title') ?: localized($page, 'title'));
        View::set('metaDescription', localized($page, 'meta_description'));
        View::set('page', $page);
        
        // Use specific template or default
        $template = $page['template'] ?: 'page';
        View::render($template);
    }
}

class ServiceController {
    public function index() {
        global $db;
        
        $services = $db->getAll("
            SELECT * FROM services 
            WHERE status = 'published' AND parent_id IS NULL
            ORDER BY sort_order
        ");
        
        View::set('pageTitle', CURRENT_LANG === 'tr' ? 'Uzmanlık Alanları' : 'Services');
        View::set('services', $services);
        
        View::render('services');
    }
    
    public function show($slug) {
        global $db;
        
        $service = $db->getRow("
            SELECT * FROM services 
            WHERE slug = ? AND status = 'published'
        ", [$slug]);
        
        if (!$service) {
            http_response_code(404);
            View::render('404');
            return;
        }
        
        // Get related services
        $relatedServices = $db->getAll("
            SELECT * FROM services 
            WHERE status = 'published' AND id != ? AND parent_id IS NULL
            ORDER BY sort_order LIMIT 4
        ", [$service['id']]);
        
        View::set('pageTitle', localized($service, 'meta_title') ?: localized($service, 'title'));
        View::set('metaDescription', localized($service, 'meta_description'));
        View::set('service', $service);
        View::set('relatedServices', $relatedServices);
        
        View::render('service-single');
    }
}

class SearchController {
    public function index() {
        global $db;
        
        $query = isset($_GET['q']) ? sanitize($_GET['q']) : '';
        $results = [];
        
        if (strlen($query) >= 3) {
            $searchTerm = '%' . $db->escapeLike($query) . '%';
            $lang = CURRENT_LANG;
            
            $results = $db->getAll("
                SELECT 'post' as type, id, slug, title_{$lang} as title, excerpt_{$lang} as excerpt, published_at as date
                FROM posts 
                WHERE status = 'published' AND (title_{$lang} LIKE ? OR content_{$lang} LIKE ?)
                UNION
                SELECT 'page' as type, id, slug, title_{$lang} as title, excerpt_{$lang} as excerpt, published_at as date
                FROM pages 
                WHERE status = 'published' AND (title_{$lang} LIKE ? OR content_{$lang} LIKE ?)
                ORDER BY date DESC
                LIMIT 50
            ", [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        }
        
        View::set('pageTitle', CURRENT_LANG === 'tr' ? 'Arama Sonuçları' : 'Search Results');
        View::set('query', $query);
        View::set('results', $results);
        
        View::render('search');
    }
}

class SitemapController {
    public function xml() {
        global $db;
        
        header('Content-Type: application/xml; charset=utf-8');
        
        echo '<?xml version="1.0" encoding="UTF-8"?>';
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
        
        // Homepage
        echo '<url><loc>' . SITE_URL . '</loc><changefreq>daily</changefreq><priority>1.0</priority></url>';
        
        // Pages
        $pages = $db->getAll("SELECT slug, updated_at FROM pages WHERE status = 'published'");
        foreach ($pages as $page) {
            echo '<url>';
            echo '<loc>' . SITE_URL . '/' . $page['slug'] . '</loc>';
            echo '<lastmod>' . date('Y-m-d', strtotime($page['updated_at'])) . '</lastmod>';
            echo '<changefreq>monthly</changefreq>';
            echo '<priority>0.8</priority>';
            echo '</url>';
        }
        
        // Blog posts
        $posts = $db->getAll("SELECT slug, updated_at FROM posts WHERE status = 'published'");
        foreach ($posts as $post) {
            echo '<url>';
            echo '<loc>' . SITE_URL . '/blog/' . $post['slug'] . '</loc>';
            echo '<lastmod>' . date('Y-m-d', strtotime($post['updated_at'])) . '</lastmod>';
            echo '<changefreq>weekly</changefreq>';
            echo '<priority>0.7</priority>';
            echo '</url>';
        }
        
        // Services
        $services = $db->getAll("SELECT slug, updated_at FROM services WHERE status = 'published'");
        foreach ($services as $service) {
            echo '<url>';
            echo '<loc>' . SITE_URL . '/uzmanlik-alanlari/' . $service['slug'] . '</loc>';
            echo '<lastmod>' . date('Y-m-d', strtotime($service['updated_at'])) . '</lastmod>';
            echo '<changefreq>monthly</changefreq>';
            echo '<priority>0.8</priority>';
            echo '</url>';
        }
        
        echo '</urlset>';
    }
}

// Run the router
$router->run();
