<?php
/**
 * View Class
 * Simple template rendering system
 */

class View {
    private static array $data = [];
    private static string $themePath = '';
    
    /**
     * Initialize theme path
     */
    public static function init(string $theme = 'cliniq'): void {
        self::$themePath = ROOT_PATH . '/themes/' . $theme . '/templates';
    }
    
    /**
     * Set data for templates
     */
    public static function set(string $key, $value): void {
        self::$data[$key] = $value;
    }
    
    /**
     * Get data from templates
     */
    public static function get(string $key, $default = null) {
        return self::$data[$key] ?? $default;
    }
    
    /**
     * Check if data exists
     */
    public static function has(string $key): bool {
        return isset(self::$data[$key]);
    }
    
    /**
     * Render a template
     */
    public static function render(string $template, array $data = []): void {
        // Merge data
        foreach ($data as $key => $value) {
            self::$data[$key] = $value;
        }
        
        // Extract data to local scope
        extract(self::$data);
        
        // Find template file
        $templateFile = self::findTemplate($template);
        
        if (!$templateFile) {
            throw new Exception("Template not found: {$template}");
        }
        
        include $templateFile;
    }
    
    /**
     * Render a partial template
     */
    public static function partial(string $partial, array $data = []): void {
        // Merge data
        foreach ($data as $key => $value) {
            self::$data[$key] = $value;
        }
        
        // Extract data to local scope
        extract(self::$data);
        
        // Find partial file
        $partialFile = self::$themePath . '/' . $partial . '.php';
        
        if (!file_exists($partialFile)) {
            // Try in partials subdirectory
            $partialFile = self::$themePath . '/partials/' . $partial . '.php';
        }
        
        if (file_exists($partialFile)) {
            include $partialFile;
        }
    }
    
    /**
     * Find template file
     */
    private static function findTemplate(string $template): ?string {
        $paths = [
            self::$themePath . '/' . $template . '.php',
            self::$themePath . '/' . $template . '/index.php',
            ROOT_PATH . '/templates/' . $template . '.php',
        ];
        
        foreach ($paths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }
        
        return null;
    }
    
    /**
     * Render JSON response
     */
    public static function json(array $data, int $statusCode = 200): void {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    /**
     * Redirect to URL
     */
    public static function redirect(string $url, int $statusCode = 302): void {
        http_response_code($statusCode);
        header('Location: ' . $url);
        exit;
    }
    
    /**
     * Show 404 page
     */
    public static function notFound(): void {
        http_response_code(404);
        self::render('404');
        exit;
    }
    
    /**
     * Get all data
     */
    public static function all(): array {
        return self::$data;
    }
    
    /**
     * Clear all data
     */
    public static function clear(): void {
        self::$data = [];
    }
}
